package formdef.plugin.config;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.beanutils.DynaBean;
import org.apache.commons.beanutils.BeanUtils;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.config.FormBeanConfig;
import org.apache.struts.config.FormPropertyConfig;
import org.apache.struts.action.DynaActionForm;

import java.util.*;

import formdef.plugin.util.FormUtils;
import formdef.plugin.FormMapping;
import formdef.plugin.conversion.ConverterFactory;


/**
 * Holds the configuration information for the FormDef engine,
 * and contains the code which generates the {@link FormMapping}
 * to {@link FormBeanConfig}.
 * <p/>
 * @author Hubert Rabago
 */
public class FormDefConfig {

    private static final Log log = LogFactory.getLog(FormDefConfig.class);

    public static final String FORMDEF_KEY = "formdef.plugin.FormDefPlugIn";

    /** Holds the form definitions */
    protected Map formDefForms = null;

    /** Holds the form configurations */
    protected Map formDefFormConfigs = null;

    /** The global property type converters */
    protected Map propertyTypeConverters = null;

    /** The global property name converters */
    protected Map propertyNameConverters = null;

    /** The global named converters */
    protected Map converterNameConverters = null;

    /** The class of the default ActionForm type to use for form beans */
    protected Class formType = null;

    /**
     * Create an instance of this object and initialize internal properties
     */
    public FormDefConfig() {
        formDefForms = new HashMap();
        formDefFormConfigs = new HashMap();
        propertyTypeConverters = new LinkedHashMap();
        propertyNameConverters = new HashMap();
        converterNameConverters = new HashMap();
        formType = DynaActionForm.class;
    }

    /**
     * Add a {@link FormMappingConfig} object to the definitions
     * held by this object
     * <p/>
     * @param formConfig the {@link FormMappingConfig} object to add
     */
    public void addForm(FormMappingConfig formConfig)
            throws Exception {
        formDefForms.put(formConfig.getName(),formConfig.generateForm(this));
        formDefFormConfigs.put(formConfig.getName(),formConfig);
    }

    /**
     * Add a {@link GlobalConverterConfig} object to the appropriate
     *      converter list.
     * <p/>
     * @param config the {@link GlobalConverterConfig} object to add.
     * @throws ClassNotFoundException if the converter is for a property type
     *      and the target Class could not be loaded.
     */
    public void addConverter(GlobalConverterConfig config)
            throws ClassNotFoundException {

        int forValue = config.getForValue();
        
        if (forValue == GlobalConverterConfig.FOR_PROPERTY_TYPE) {
            Class targetClass = FormUtils.getInstance().classForName(config.getTarget());
            config.setTargetClass(targetClass);
            propertyTypeConverters.put(config.getTarget(),config);

        } else if (forValue == GlobalConverterConfig.FOR_PROPERTY_NAME) {
            propertyNameConverters.put(config.getTarget(),config);

        } else if (forValue == GlobalConverterConfig.FOR_CONVERTER_NAME) {
            converterNameConverters.put(config.getTarget(),config);
        }
    }

    /**
     * Specify the name of the class to use as the default ActionForm type
     * for generating struts form beans.
     * <p/>
     * @param formTypeName the name of the ActionForm class to use
     */
    public void setFormType(String formTypeName)
            throws ClassNotFoundException {
        if (log.isInfoEnabled()) {
            log.info("Using " + formTypeName + " as default form type.");
        }
        formType = Class.forName(formTypeName);
    }

    /**
     * Specify the name of the class to use as FormUtilsFactory.
     * <p/>
     * @param formUtilsFactory the name of the FormUtilsFactory instance
     * to use.
     */
    public void setFormUtilsFactory(String formUtilsFactory) {
        if (log.isInfoEnabled()) {
            log.info("Using " + formUtilsFactory + " as FormUtils factory.");
        }
        FormUtils.setFactoryClassName(formUtilsFactory);
    }

    /**
     * Specify the name of the class to use as ConverterFactory.
     * <p/>
     * @param converterFactory the name of the ConverterFactory instance
     * to use.
     */
    public void setConverterFactory(String converterFactory) {
        if (log.isInfoEnabled()) {
            log.info("Using " + converterFactory + " as ConverterFactory.");
        }
        ConverterFactory.setConverterFactoryClassName(converterFactory);
    }

    /**
     * Find the property type converter that can be used to convert the
     * given propertyType.  A converter that is specifically assigned to handle
     * the given type is first sought; if none is found, those that do not
     * require an exact type match are checked.
     * <p/>
     * @param propertyType the Class of the property to find a converter for
     * @return the {@link GlobalConverterConfig} of the converter assigned to
     * the given type.
     */ 
    public GlobalConverterConfig findPropertyTypeConverter(Class propertyType) {
        GlobalConverterConfig result = (GlobalConverterConfig)
                propertyTypeConverters.get(propertyType.getName());

        // did we get an exact match?
        if (result == null) {
            // we didn't find an exact match for the specified property type.
            //  try to look for an compatible converter
            Iterator iter = propertyTypeConverters.keySet().iterator();
            while (iter.hasNext()) {
                GlobalConverterConfig config = (GlobalConverterConfig)
                        propertyTypeConverters.get(iter.next());

                // can this config be used for compatible class types?
                if (config.isExactMatch()) {
                    continue;
                }

                if (config.getTargetClass() != null) {
                    if (config.getTargetClass()
                            .isAssignableFrom(propertyType)) {
                        return config;
                    }
                }
            }
        }
        return result;
    }

    /**
     * Find the configuration for the converter assigned to handle properties
     * with the given name.
     * <p/>
     * @param propertyName the name of the property to find a converter for
     * @return the {@link GlobalConverterConfig} of the converter assigned to
     * the given property name.
     */ 
    public GlobalConverterConfig findPropertyNameConverter(String propertyName) {
        return (GlobalConverterConfig) propertyNameConverters.get(propertyName);
    }

    /**
     * Find the configuration for the converter with the specified converter 
     * name.
     * <p/>
     * @param converterName the name used to identify the converter
     * @return the {@link GlobalConverterConfig} of the converter with the
     * given name
     */ 
    public GlobalConverterConfig findConverterNameConverter(String converterName) {
        return (GlobalConverterConfig) converterNameConverters.get(converterName);
    }

    ///**
    // * Used for debugging purposes only
    // */
    ///* package */ Map getFormDefForms() {
    //    return formDefForms;
    //}

    /**
     * Get the names of the forms defined in this object.
     * <p/>
     * @return an array of string containing the form names.
     */
    public String[] getFormNames() {
        ArrayList names = new ArrayList();
        Set keys = formDefForms.keySet();
        Iterator iter = keys.iterator();
        while (iter.hasNext()) {
            FormMapping form = (FormMapping) formDefForms.get(iter.next());
            names.add(form.getName());
        }
        
        return (String[]) names.toArray(new String[names.size()]);
    }

    /**
     * Get the {@link FormMapping} definition with the given name.
     * <p/>
     * @param name the name of the form to get
     * @return the form definition with the given name
     */
    public FormMapping getForm(String name) {
        return (FormMapping) formDefForms.get(name);
    }

    /**
     * Register the forms in Struts.
     * <p/>
     * @param moduleConfig the {@link ModuleConfig} of the module being configured
     */
    public void registerForms(ModuleConfig moduleConfig)
            throws ClassNotFoundException {
        Set keys = formDefForms.keySet();
        Iterator iter = keys.iterator();
        while (iter.hasNext()) {
            FormMapping form = (FormMapping) formDefForms.get(iter.next());
            FormBeanConfig formBeanConfig =
                    generateFormBeanConfig(form,moduleConfig);
            if (log.isTraceEnabled()) {
                log.trace("adding formBeanConfig: " + formBeanConfig);
            }
            moduleConfig.addFormBeanConfig(formBeanConfig);
            form.setFormBeanConfig(formBeanConfig);
        }
    }

    /**
     * Generate a {@link FormBeanConfig} for this form definition.  The generated
     * object can be added to the Struts configuration for use in Struts forms.
     * <p/>
     * @return the {@link FormBeanConfig} for this form definition
     * @throws ClassNotFoundException if the type for this form cannot be found
     */
    public FormBeanConfig generateFormBeanConfig(FormMapping form,
                                                 ModuleConfig moduleConfig)
            throws ClassNotFoundException {

        // create a FormBeanConfig instance with the fields we gathered
        FormBeanConfig formBeanConfig = new FormBeanConfig();
        formBeanConfig.setName(form.getName());
        
        // Struts1.1 - uncomment the line below to build a Struts 1.1-compatible version
        //formBeanConfig.setModuleConfig(moduleConfig);

        // check what type of ActionForm we'll use for this bean
        String formBeanType = getFormType(form.getName());
        if (log.isTraceEnabled()) {
            log.trace(form.getName() + ": using " + formBeanType);
        }
        formBeanConfig.setType(formBeanType);

        // is this an action form that's not a dyna form?
        if (formBeanType != null) {
            Class formBeanClass = Class.forName(formBeanType);
            if (!DynaBean.class.isAssignableFrom(formBeanClass)) {
                return formBeanConfig;
            }
        }

        FormMappingConfig formConfig =
                (FormMappingConfig) formDefFormConfigs.get(form.getName());

        // loop thru the properties in the form mapping 
        //      and create the FormPropertyConfig objects
        Iterator iter = form.getProperties().keySet().iterator();
        while (iter.hasNext()) {
            String propertyName = (String) iter.next();
            FormPropertyConfig propConfig =
                    createFormPropertyConfig(propertyName, formConfig);

            formBeanConfig.addFormPropertyConfig(propConfig);
        }
        return formBeanConfig;
    }


    /**
     * Create the Struts FormPropertyConfig for the named property.
     * <p/>
     * @param propertyName the name of the property to process
     * @param formConfig the form configuration of the named property
     * @return a FormPropertyConfig for use in defining Struts form beans
     */
    protected FormPropertyConfig createFormPropertyConfig(
            String propertyName,
            FormMappingConfig formConfig) {
        if (log.isTraceEnabled()) {
            log.trace("property: [" + propertyName + "]");
        }

        // get the PropertyMappingConfig for this prop
        PropertyMappingConfig config = formConfig.getProperty(propertyName);

        // create and configure the return object
        FormPropertyConfig propConfig =
                new FormPropertyConfig();
        propConfig.setName(propertyName);

        // the type defaults to a String; it can be overridden by specifying
        //  a type value, or a formName, in which case, the type is the
        //  formType of the named form.
        if (config.getType() != null) {
            propConfig.setType(config.getType());
        } else if (config.getFormName() != null) {
            propConfig.setType(getFormType(config.getFormName()));
        } else {
            propConfig.setType(String.class.getName());
        }

        if (config.getInitial() != null) {
            propConfig.setInitial(config.getInitial());
        }
        if (config.getSize() != null) {
            propConfig.setSize(Integer.parseInt(config.getSize()));
        }
        
        // Set the 1.3.x "reset" property by reflection, if available.
        if (config.getReset() != null) {
            setByReflection("reset", propConfig, config.getReset());
        }

        return propConfig;
    }

    
    /**
     * <p>Sets the named field on the destination object using the given value,
     * all through reflection.</p>
     * 
     * @param name          the name of the object property to set.
     * @param destination   the target object whose property will be set.
     * @param value         the value to use in setting the property.
     * 
     * @return true if the set operation succeeded.
     */ 
    private boolean setByReflection(String name, 
                                    Object destination, 
                                    Object value) {
        try {
            BeanUtils.setProperty(destination, name, value);
            return true;
        } catch (Exception e) {
            return false;
        }
    }


    /**
     * Get the form type of the named form.  This method uses the 
     * information in {@link #formDefFormConfigs} and therefore should not
     * be used after {@link #cleanup()} has been called.
     * <p/>
     * 
     * @param formName the name of the form whose type is to be returned.
     * @return the name of the ActionForm subclass configured for formName. 
     */ 
    protected String getFormType(String formName) {
        FormMappingConfig formConfig =
                (FormMappingConfig) formDefFormConfigs.get(formName);
        if (formConfig == null) {
            throw new IllegalArgumentException("Unable to locate "
                + formName + " among the forms registered with FormDef for "
                + "this module.");
        }

        String formBeanType = formConfig.getFormType();
        if (formBeanType == null) {
            formBeanType = formType.getName();
        }

        return formBeanType;
    }


    /**
     * Perform cleanup work to release objects that will no longer
     *      be used after FormDef has been fully configured.
     */
    public void cleanup() {
        this.formDefFormConfigs = null;
        this.formType = null;

        // remove formDefForms that are null or have no associated bean types
        Iterator iter = this.formDefForms.keySet().iterator();
        while (iter.hasNext()) {
            String formName = (String) iter.next();
            FormMapping form = (FormMapping) this.formDefForms.get(formName);
            if ((form == null) || (form.getBeanType() == null)) {
                if (log.isDebugEnabled()) {
                    log.debug("removing " + form + " from FormDef list");
                }
                iter.remove();
            }
        }
    }

    public String toString() {
        StringBuffer result = new StringBuffer(256);
        result.append("FormDefConfig [");
        result.append("formDefForms=[").append(formDefForms).append("];");
        result.append("]");
        return result.toString();
    }

}
