package formdef.plugin.conversion;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;

import java.util.Locale;
import java.util.Date;
import java.text.SimpleDateFormat;
import java.text.ParseException;

/**
 * {@link Converter} implementation for {@link java.util.Date}, 
 * {@link java.sql.Date}, and {@link java.sql.Timestamp} objects.
 */
public class DateConverter implements Converter {
    
    private static final Log log = LogFactory.getLog(DateConverter.class);

    /**
     * <p>Converts Date values to and from Strings.</p>
     * 
     * @param context the {@link ConversionContext} holding information
     * about the value to convert and additional conversion parameters
     * @return the equivalent of the given value, in the specified type
     */
    public Object convert(ConversionContext context) {
        Object value = context.getValue();
        Class type = context.getType();
        Object param = context.getParam();
        Locale locale = context.getLocale();
        String propertyName = context.getPropertyName();
        
        if (log.isTraceEnabled()) {
            log.trace("converting [" + value + "] to type [" + type + "] using param=[" + param + "]");
        }
        
        if (param == null) {
            throw new IllegalArgumentException("Date fields require a conversion format."); 
        }
        
        // create our formatter object
        SimpleDateFormat sdf = new SimpleDateFormat((String) param, locale);
        
        // are we converting from String values to Date values?
        if (value instanceof String) {
                                    
            // create a date instance and set its value
            java.util.Date date = null;
            try {
                date = new java.util.Date();
                date.setTime(sdf.parse((String) value).getTime());
            } catch (ParseException e) {
                return null;
            }

            if (type.equals(java.util.Date.class)) {
                return date;
            } else if (type.equals(java.sql.Date.class)) {
                return new java.sql.Date(date.getTime());
            } else if (type.equals(java.sql.Timestamp.class)) {
                return new java.sql.Timestamp(date.getTime());
            } else {
                throw new IllegalArgumentException(
                        "Unsupported type parameter: " + type
                        + " for property " + propertyName);
            }
        } else {
            // either we're expected to return a string, or we were given a null
            if ((value == null) || ((value instanceof String) && (((String)value).length() <= 0))) {
                if (type.equals(String.class)) {
                    return "";
                } else {
                    return null;
                }
            }
            
            // use the formatter object to format our result
            return sdf.format((Date) value);
        }
    }



}
