package formdef.plugin.conversion;

import formdef.plugin.util.FormUtils;
import org.apache.struts.action.Action;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionMapping;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.Collection;


/**
 * <p>Abstract base {@link Converter} for use in converting between a 
 * collection of nested beans and a collection of nested forms.</p>
 * 
 * <p>A concrete subclass should provide implementations for 
 * {@link #getFormCollection} and {@link #getBeanCollection}, and 
 * may optionally override {@link #setFormValue} and {@link #setBeanValue}.</p>   
 */
public abstract class FormCollectionConverter
        implements Converter {
    
    private static final Log log = 
            LogFactory.getLog(FormCollectionConverter.class);
    
    /**
     * Key used with conversion properties to pass a flag that states whether
     * this class is being used in form initialization. See
     * {@link #getFormCollection(ConversionContext, String, Collection)}. 
     */ 
    public static final String PROP_INITIALIZE_FLAG_KEY =
            "formdef.plugin.conversion.FormCollectionConverter.initializeForm";

    
    /**
     * <p>Converts an ActionForm to a typed bean and vice-versa using
     * FormUtils.</p>     
     * 
     * @param context the {@link ConversionContext} holding information
     *                about the value to convert and additional conversion parameters
     * @return the equivalent of the given value, in the specified type
     */
    public Object convert(ConversionContext context)
            throws IllegalAccessException, InvocationTargetException {

        Object source = context.getSource();
        String propertyName = context.getPropertyName();
        Collection collection = (Collection) context.getValue();
        Object param = context.getParam();
        Class type = context.getType();
        Action action = context.getAction();
        ActionMapping mapping = context.getMapping();
        HttpServletRequest request = context.getRequest();
        
        if (log.isTraceEnabled()) {
            log.trace("converting [" + propertyName + "] to type [" + type + "] using param=[" + param + "]");
        }
        
        Collection result = null;
        
        // are we converting from an ActionForm to a business bean?
        if (source instanceof ActionForm) {
            result = getBeanCollection(context, propertyName, collection);
            
            if (collection != null) {
                // loop through the forms and convert them to business objects
                Iterator iterator = collection.iterator();
                while (iterator.hasNext()) {
                    ActionForm form = (ActionForm) iterator.next();
                    if (form == null) {
                        continue;
                    }
                    Object value = FormUtils.getFormValues(form, (String) param,
                            action, mapping, request);
                    setBeanValue(context, result, form, value);
                }
            }
        } else {
            result = getFormCollection(context, propertyName, collection);
            
            if (collection != null) {
                // loop through the business objects and convert them to forms
                Iterator iterator = collection.iterator();
                while (iterator.hasNext()) {
                    Object bean = iterator.next();
                    if (bean == null) {
                        continue;
                    }
                    ActionForm form = FormUtils.setFormValues((String) param,
                        bean, action, mapping, request);
                    setFormValue(context, result, bean, form);
                }
            }
        }
        
        return result;
    }

    
    /**
     * <p>Adds the given ActionForm to the collection as appropriate.  
     * Subclasses can override this method to control how the form
     * collection is populated.</p>
     * 
     * @param context   the context object containing parameters used for
     *                  the conversion.
     * @param collection    the collection of ActionForms being used 
     *                      for the field identified by the context's
     *                      propertyName value. 
     * @param source        the business object used to create the form.
     * @param value         the form being added to the collection.
     */ 
    public void setFormValue(ConversionContext context,
                             Collection collection,
                             Object source,
                             ActionForm value) {
        collection.add(value);
    }
    
    
    /**
     * <p>Adds the given business object value to the collection as appropriate.
     * Subclasses can override this method to control how the bean
     * collection is populated.</p>
     * 
     * @param context   the context object containing parameters used for
     *                  the conversion.
     * @param collection    the collection of business objects holding the 
     *                      data for the field identified by the context's
     *                      propertyName value. 
     * @param source        the ActionForm used to create the value.
     * @param value         the value being added to the collection.
     */ 
    public void setBeanValue(ConversionContext context,
                             Collection collection,
                             ActionForm source,
                             Object value) {
        collection.add(value);
    }


    //******************************************************** Abstract methods
    
    
    /**
     * <p>Create and return the Collection that will be used to hold the nested
     * ActionForm objects that will be used for the field identified by 
     * <code>propertyName</code>.</p>
     * 
     * <p>This method may also be called to create a collection to be used in 
     * initializing a field of a collection of forms for an empty form.  When
     * this is the case, a String passed through the context's properties, 
     * using the {@link #PROP_INITIALIZE_FLAG_KEY} key, will be set to "true".  
     * Also, the given <code>context</code>'s <code>source</code> value, 
     * as well as the <code>source</code> value being passed in as parameter
     * to this method, will be null.  </p>
     * 
     * <p>A basic implementation can simply return an empty collection.</p>
     *  
     * @param context   the context object containing parameters used for
     *                  the conversion.
     * @param propertyName  the name of the ActionForm field whose collection 
     *                      is needed.
     * @param source    the value of the named property from the business
     *                  object.  May be null if this method is being called
     *                  to create an empty form.    
     * @return  the collection object that will be used to hold nested 
     *  ActionForm object for the named property.
     */ 
    public abstract Collection getFormCollection(ConversionContext context,
                                                 String propertyName,
                                                 Collection source);
    
    
    /**
     * <p>Create and return the Collection that will be used to hold the
     * business objects that will be used for the field identified by 
     * <code>propertyName</code>.</p>
     * 
     * <p>A basic implementation can simply return an empty collection.</p>
     * 
     * @param context   the context object containing parameters used for
     *                  the conversion.
     * @param propertyName  the name of the business object field whose 
     *                      collection is needed.
     * @param source    the value of the named property from the business
     *                  object. 
     * @return  the collection object that will be used to hold nested 
     *  ActionForm object for the named property.
     */ 
    public abstract Collection getBeanCollection(ConversionContext context,
                                                 String propertyName,
                                                 Collection source);


}
