package formdef.plugin.conversion;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;

import java.text.ParseException;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.Locale;

/**
 * {@link Converter} implementation for the primitive numeric types
 * and their wrapper objects.
 */
public class NumberConverter implements Converter {

    private static final Log log = LogFactory.getLog(NumberConverter.class);

    /**
     * <p>Converts the value to the specified type, using a
     * {@link DecimalFormat} to handle output formatting or parsing of
     * formatted input values.</p>
     *
     * @param context the {@link ConversionContext} holding information
     * about the value to convert and additional conversion parameters
     * @return the equivalent of the given value, in the specified type
     */
    public Object convert(ConversionContext context) {
        Object value = context.getValue();
        Class type = context.getType();
        Object param = context.getParam();
        String propertyName = context.getPropertyName();

        if (log.isTraceEnabled()) {
            log.trace("converting [" + value + "] to type [" + type + "] using param=[" + param + "]");
        }
        if (param == null) {
            if ((value == null) || ((value instanceof String) && (((String)value).length() <= 0))) {
                if ((type.equals(Integer.TYPE))) {
                    return new Integer(0);
                } else if (type.equals(Long.TYPE)) {
                    return new Long(0);
                } else if  (type.equals(Double.TYPE)) {
                    return new Double(0);
                } else if  (type.equals(Float.TYPE)) {
                    return new Float(0);
                } else if (type.equals(Byte.TYPE)) {
                    return new Byte((byte)0);
                } else if (type.equals(Short.TYPE)) {
                    return new Short((short)0);
                } else
                     if ((type.equals(String.class)) ||
                         (type.equals(Integer.class)) ||
                         (type.equals(Long.class)) ||
                         (type.equals(Double.class)) ||
                         (type.equals(Float.class)) ||
                         (type.equals(Byte.class)) ||
                         (type.equals(Short.class)))
                         {
                           return null;
                         } else {
                         throw new IllegalArgumentException(
                                 "Unsupported type parameter: " + type
                                 + " for property " + propertyName);
                }
            }

            if ((type.equals(Integer.class)) || (type.equals(Integer.TYPE))) {
                return Integer.valueOf((String)value);
            } else if ((type.equals(Long.class)) || (type.equals(Long.TYPE))) {
                return Long.valueOf((String)value);
            } else if ((type.equals(Double.class)) || (type.equals(Double.TYPE))) {
                return Double.valueOf((String)value);
            } else if ((type.equals(Float.class)) || (type.equals(Float.TYPE))) {
                return Float.valueOf((String)value);
            } else if ((type.equals(Byte.class)) || (type.equals(Byte.TYPE))) {
                return Byte.valueOf((String)value);
            } else if ((type.equals(Short.class)) || (type.equals(Short.TYPE))) {
                return Short.valueOf((String)value);
            } else if (type.equals(String.class)) {
                return value.toString();
            } else {
                throw new IllegalArgumentException(
                        "Unsupported type parameter: " + type
                        + " for property " + propertyName);
            }
        }

        // create a DecimalFormat applicable to our conversion context
        DecimalFormat df = createDecimalFormat(context);

        if (value instanceof String) {
            if (value == null) {
                return null;
            }
            try {
                //return df.parse((String) value);
                if ((type.equals(Integer.class)) || (type.equals(Integer.TYPE))) {
                    return new Integer(df.parse((String) value).intValue());
                } else if ((type.equals(Long.class)) || (type.equals(Long.TYPE))) {
                    return new Long(df.parse((String) value).longValue());
                } else if ((type.equals(Double.class)) || (type.equals(Double.TYPE))) {
                    return new Double(df.parse((String) value).doubleValue());
                } else if ((type.equals(Float.class)) || (type.equals(Float.TYPE))) {
                    return new Float(df.parse((String) value).floatValue());
                } else if ((type.equals(Byte.class)) || (type.equals(Byte.TYPE))) {
                    return new Byte(df.parse((String) value).byteValue());
                } else if ((type.equals(Short.class)) || (type.equals(Short.TYPE))) {
                    return new Short(df.parse((String) value).shortValue());
                } else {
                    throw new IllegalArgumentException(
                            "Unsupported type parameter: " + type
                            + " for property " + propertyName);
                }
            } catch (ParseException e) {
                return null;
            }
        } else {
            if (value == null) {
                return "";
            }
            return df.format(value);
        }
    }

    /**
     * Create a DecimalFormat instance to be used in performing the conversion provided
     * in the given context object
     * @param context the details of the conversion for which 
     * @return
     */
    protected DecimalFormat createDecimalFormat(ConversionContext context) {
        Locale locale = context.getLocale();
        Object param = context.getParam();

        // create a decimal format specific to the locale being used
        DecimalFormat df = (DecimalFormat) NumberFormat.getNumberInstance(locale);

        // apply the conversion parameter
        df.applyLocalizedPattern((String) param);
        return df;
    }

}
