package formdef.plugin.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Properties;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.config.ModuleConfig;

import javax.servlet.ServletContext;

import formdef.plugin.config.FormDefConfig;
import formdef.plugin.FormMapping;
import formdef.plugin.PropertyMapping;

/**
 * Implementation of {@link ResourceLocaleUtil} that will return a Locale
 * to match a conversion key given to PropertyMessageResources.
 * <p/>
 * This class uses code based on PropertyMessageResources, and also
 * caches all message keys for faster future retrieval. 
 * <p/>
 */
public class PropertyResourceLocaleUtil 
        implements ResourceLocaleUtil {


    // ------------------------------------------------------------- Properties


    /**
     * The default Locale for our environment.
     */
    protected Locale defaultLocale = Locale.getDefault();

    protected String config = null;

    /**
     * The set of locale/keys combinataions which we have already processed, 
     * keyed by the value calculated in {@link #localeKey(Locale)}.
     */
    protected HashMap locales = new HashMap();


    /**
     * The <code>Log</code> instance for this class.
     */
	protected static final Log log =
		LogFactory.getLog(PropertyResourceLocaleUtil.class);

    /**
     * The list of FormDef keys for this util's module.
     */ 
    protected List keys = new ArrayList();

    /**
     * The cache of locale results we have accumulated over time, keyed by the
     * value calculated in <code>formResultKey()</code>.
     */
    protected HashMap localeResults = new HashMap();


    // ----------------------------------------------------------- Constructors


    /**
     * Create an instance of this object that will be used to determine the
     * locales to be used for keys provided in the resource file specified 
     * in the config string.
     * @param context   the servlet context for the application.
     * @param moduleConfig  the struts config for the current module.
     * @param config    The configuration parameter passed to the resource
     *                  object.
     */ 
    public PropertyResourceLocaleUtil(ServletContext context,
                                      ModuleConfig moduleConfig,
                                      String config) {

        log.trace("Initializing, config='" + config + "'");
        this.config = config;

        // load the keys in this module's FormDefConfig
        FormDefConfig formDefConfig = 
                FormUtils.getInstance().getFormDefConfig(context, moduleConfig);
        loadKeys(formDefConfig);
    }



    // --------------------------------------------------------- Public Methods


    /**
     * Finds the Locale appropriate for the message returned when the given
     * conversionKey is looked up for the given locale. 
     * <p/>
     * @param locale The requested message Locale, or <code>null</code>
     *  for the system default Locale.
     * @param conversionKey The message key to look up.
     * @return the actual locale that the key has a value for.
     */
    public Locale getLocale(Locale locale, String conversionKey) {
        
        if (log.isDebugEnabled()) {
            log.debug("getLocale(" + locale + "," + conversionKey + ")");
        }

        // Initialize variables we will require
        String localeString = localeKey(locale);
        
        String resultKey = null;
        String resultLocale = null;
        int underscore = 0;
        boolean addIt = false;  // Add if not found under the original key

        // originalKey retains the key we started with, before any processing.
        //  whatever result we get gets cached under the original key, even
        //  though we might generalize the locale to resolve it.
        String originalKey = formResultKey(localeString, conversionKey);
        
        // Loop from specific to general Locales looking for our conversion key
        while (true) {

            // Load this Locale's messages if we have not done so yet
            loadLocale(localeString);
            
            // form the key that we use for the current locale
            resultKey = formResultKey(localeString, conversionKey);
            
            synchronized (localeResults) {
                // Do we already know the locale to use for this locale/key combo?
                resultLocale = (String) localeResults.get(resultKey);
                
                if (resultLocale != null) {
                    // we have our result! now let's cache it using the 
                    //      original input locale.
                    if (addIt) {
                        localeResults.put(originalKey, resultLocale);
                    }
                    
                    log.debug("returning locale " + localeString);
                    // return a Locale object for the resulting locale 
                    return createLocale(resultLocale);
                }
            }
            
            // if we reach this point, this means we didn't find an entry for
            //  the original/given locale, so whatever we find beyond this point
            //  should be added to the cache
            addIt = true;

            // strip trailing modifiers to try a more general locale conversionKey
            underscore = localeString.lastIndexOf("_");
            if (underscore < 0) {
                break;
            }
            localeString = localeString.substring(0, underscore);

        }

        // try the default locale if the current locale is different
        if (!defaultLocale.equals(locale)) {
            // do essentially the same stuff as we did above
            localeString = localeKey(defaultLocale);
            resultKey = formResultKey(localeString, conversionKey);
            loadLocale(localeString);
            synchronized (localeResults) {
                resultLocale = (String) localeResults.get(resultKey);
                if (resultLocale != null) {
                    localeResults.put(originalKey, resultLocale);
                    log.debug("returning defaultLocale " + defaultLocale);
                    return defaultLocale;
                }
            }
        }

        // As a last resort, try the empty localeString
        localeString = "";
        resultKey = formResultKey(localeString, conversionKey);
        loadLocale(localeString);
        synchronized (localeResults) {
            resultLocale = (String) localeResults.get(resultKey);
            if (resultLocale != null) {
                // if the key is present, then we can use the default locale
                localeResults.put(originalKey, Locale.getDefault().toString());
                log.debug("returning Locale.getDefault() " + Locale.getDefault());
                return Locale.getDefault();
            }
        }

        // nothing found
        log.debug("returning null locale ");
        return null;

    }

    /**
     * Create a locale object that matches the given key.
     * @param localeKey a string matching what {@link Locale#toString()} would
     *      provide
     * @return the key matching the given key
     */ 
    protected Locale createLocale(String localeKey) {
        String language = localeKey;
        String variant = "";
        String country = "";

        int underscore = localeKey.indexOf("_");
        if (underscore >= 0) {
            language = localeKey.substring(0, underscore);
            localeKey = localeKey.substring(underscore+1);
            country = localeKey;
            underscore = localeKey.indexOf("_");
            if (underscore >= 0) {
                country = localeKey.substring(0, underscore);
                localeKey = localeKey.substring(underscore+1);
                variant = localeKey;
            }
        }

        return new Locale(language, country, variant);
    }

    
    // ------------------------------------------------------ Protected Methods


    /**
     * Load the messages associated with the specified Locale key.  For this
     * implementation, the <code>config</code> property should contain a fully
     * qualified package and resource name, separated by periods, of a series
     * of property resources to be loaded from the class loader that created
     * this PropertyMessageResources instance.  This is exactly the same name
     * format you would use when utilizing the
     * <code>java.util.PropertyResourceBundle</code> class.
     *
     * @param localeKey Locale key for the messages to be retrieved
     */
    protected synchronized void loadLocale(String localeKey) {

        if (log.isTraceEnabled()) {
            log.trace("loadLocale(" + localeKey + ")");
        }
        
        // Have we already attempted to load messages for this locale?
        if (locales.get(localeKey) != null) {
            return;
        }
        
        locales.put(localeKey, localeKey);

        // Set up to load the property resource for this locale key, if we can
        String name = config.replace('.', '/');
        if (localeKey.length() > 0) {
            name += "_" + localeKey;
        }
        
        name += ".properties";
        InputStream is = null;
        Properties props = new Properties();

        // Load the specified property resource
        if (log.isTraceEnabled()) {
            log.trace("  Loading resource '" + name + "'");
        }
        
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if (classLoader == null) {
            classLoader = this.getClass().getClassLoader();
        }
        
        is = classLoader.getResourceAsStream(name);
        if (is != null) {
            try {
                props.load(is);
                
            } catch (IOException e) {
                log.error("loadLocale()", e);
            } finally {
                try {
                    is.close();
                } catch (IOException e) {
                    log.error("loadLocale()", e);
                }
            }
        }
        
        if (log.isTraceEnabled()) {
            log.trace("  Loading resource completed");
        }

        // Copy the corresponding values into our cache
        if (props.size() < 1) {
            return;
        }
        
        synchronized (localeResults) {
            // try to find the conversion keys within the props we just loaded
            for (Iterator iterator = keys.iterator(); iterator.hasNext();) {
                String conversionKey = (String) iterator.next();
                
                // if this key is in the props we read, take note of that in localeResults
                if (props.get(conversionKey) != null) {
                    String resultKey = formResultKey(localeKey, conversionKey);
                    if (log.isTraceEnabled()) {
                        log.trace("  Saving result key '" + resultKey);
                    }
                    
                    // cache the locale key for the result key we're holding
                    localeResults.put(resultKey, localeKey);
                }
            }
        }

    }



    /**
     * Compute and return a key to be used in caching information by a Locale.
     * <strong>NOTE</strong> - The locale key for the default Locale in our
     * environment is a zero length String.
     *
     * @param locale The locale for which a key is desired
     */
    protected String localeKey(Locale locale) {
        return (locale == null) ? "" : locale.toString();
    }

    /**
     * Compute and return a key to be used in caching information
     * by Locale and message key.
     *
     * @param locale The Locale for which this format key is calculated
     * @param key The message key for which this format key is calculated
     */
    protected String messageKey(Locale locale, String key) {

        return (localeKey(locale) + "." + key).intern();

    }


    /**
     * Compute and return a key to be used in caching information
     * by locale key and message key.
     * <p/>
     * @param localeKey The locale key for which this cache key is calculated
     * @param key The message key for which this cache key is calculated
     */
    protected String formResultKey(String localeKey, String key) {

        return (localeKey + "." + key).intern();

    }
    
    
    /**
     * Load the resource keys used by the forms configured for 
     * the given module.
     */ 
    protected void loadKeys(FormDefConfig formDefConfig) {
        // get the names of all the forms configured with formDefConfig
        String[] formNames = formDefConfig.getFormNames();

        // loop through all the forms and get their conversion keys
        for (int i = 0; i < formNames.length; i++) {
            FormMapping formMapping = formDefConfig.getForm(formNames[i]);
            
            // loop through the properties and keep track of any 
            //      conversion keys
            Map properties = formMapping.getProperties();
            Set propKeys = properties.keySet();
            for (Iterator iterator = propKeys.iterator(); iterator.hasNext();) {
                PropertyMapping property = (PropertyMapping) properties.get(iterator.next());
                
                // get the conversion key for this prop
                String conversionKey = property.getConversionKey();
                if (conversionKey != null) {
                    // keep it in our instance list
                    keys.add(conversionKey);
                }
            }
        }
    }
    
}
