package formdef.plugin;

import org.apache.struts.action.PlugIn;
import org.apache.struts.action.ActionServlet;
import org.apache.struts.config.ModuleConfig;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;

import javax.servlet.ServletException;
import java.io.IOException;
import java.io.InputStream;
import java.io.BufferedInputStream;
import java.util.StringTokenizer;

import formdef.plugin.digester.DefaultDigester;
import formdef.plugin.config.FormDefConfig;

/**
 * Read the FormDef configuration files and initialize the FormDef engine.
 * <p/>
 *
 * @author Hubert Rabago
 */
public class FormDefPlugIn implements PlugIn {

    private static final Log log = LogFactory.getLog(FormDefPlugIn.class);

    /**
     * Delimitter for Validator resources.
     */
    private static final String RESOURCE_DELIM = ",";

    protected ActionServlet servlet = null;
    protected ModuleConfig moduleConfig = null;
    protected FormDefConfig formDefConfig = null;
    protected String pathnames = null;
    protected String defnames = null;

    //********************************
    // PlugIn interface implementation methods

    /**
     * <p>Receive notification that our owning module is being
     * shut down.</p>
     */
    public void destroy() {
    }

    /**
     * <p>Receive notification that the specified module is being
     * started up.</p>
     *
     * @param servlet ActionServlet that is managing all the
     *                modules in this web application
     * @param config  ModuleConfig for the module with which
     *                this plug-in is associated
     * @throws ServletException if this <code>PlugIn</code> cannot
     *                          be successfully initialized
     */
    public void init(ActionServlet servlet, ModuleConfig config)
            throws ServletException {

        this.servlet = servlet;
        moduleConfig = config;
        formDefConfig = new FormDefConfig();

        try {
            // process the defnames
            loadConfig(formDefConfig);
        } catch (IOException e) {
            throw new ServletException("Unable to configure FormDef plugin", e);
        }

        try {
            formDefConfig.registerForms(moduleConfig);
        } catch (ClassNotFoundException e) {
            throw new ServletException(e);
        }
        formDefConfig.cleanup();
        servlet.getServletContext()
                .setAttribute(FormDefConfig.FORMDEF_KEY + config.getPrefix(),
                        formDefConfig);
    }

    //********************************
    // accessors for plugin parameters

    public String getPathnames() {
        return pathnames;
    }

    public void setPathnames(String pathnames) {
        this.pathnames = pathnames;
    }

    public String getDefnames() {
        return defnames;
    }

    public void setDefnames(String defnames) {
        this.defnames = defnames;
    }

    //********************************
    //

    protected void loadConfig(FormDefConfig formDefConfig)
            throws IOException {

        if (defnames != null && defnames.length() > 0) {
            loadConfigFromParameter(defnames, formDefConfig);
        }

        if (pathnames != null && pathnames.length() > 0) {
            loadConfigFromParameter(pathnames, formDefConfig);
        }

    }

    protected void loadConfigFromParameter(String filenames,
                                         FormDefConfig formDefConfig)
            throws IOException {
        log.trace("in loadConfigFromParameter where filenames=" + filenames);
        StringTokenizer st = new StringTokenizer(filenames, RESOURCE_DELIM);
        while (st.hasMoreTokens()) {
            String formDefFilename = st.nextToken().trim();

            if (log.isInfoEnabled()) {
                log.info("Loading form definition file '"
                        + formDefFilename + "'");
            }

            InputStream input =
                    servlet.getServletContext().getResourceAsStream(
                            formDefFilename);

            if (input != null) {
                BufferedInputStream bis = new BufferedInputStream(input);

                try {
                    DefaultDigester.readConfiguration(formDefConfig, bis);
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                } finally {
                    bis.close();
                }

            } else {
                log.error("Skipping form definition file from '"
                        + formDefFilename
                        + "'.  No stream could be opened.");
            }
        }
    }

}
