package formdef.plugin;

import formdef.plugin.conversion.FormConverter;

import java.lang.reflect.Method;
import java.util.Properties;

/**
 * <p>Holds typed information about a property in a bean used for processing
 * forms at runtime.</p>
 * 
 * @author Hubert Rabago
 */
public class PropertyMapping {

    //****************************************************** Instance variables
    
    /** 
     * The name of the property on the form bean. 
     */
    private String name;

    /**
     * The class of the property on the business object.
     */ 
    private Class type;

    /**
     * The name of the form bean that will hold the data
     * for this property.
     */ 
    private String formName;

    /**
     * The getter method of the business object to use for this property.
     */ 
    private Method getter;

    /**
     * The setter method of the business object to use for this property.
     */ 
    private Method setter;
    
    /**
     * The converter object to use for this property.
     */ 
    private Object converter;
    
    /**
     * The conversion parameter to use when converting this property.
     */ 
    private String conversionParam;
    
    /**
     * The resource key that points to the conversion parameter for this 
     * property.
     */ 
    private String conversionKey;
    
    /**
     * The name of the resource bundle where the conversion parameter is 
     * located.
     */ 
    private String conversionBundle;
    
    /**
     * The default value to use for this property.
     */ 
    private String defaultValue;
    
    /**
     * Arbitrary properties that are configured for this form property.
     */ 
    private Properties properties;
    
    /**
     * The Class used for this field in the form bean.
     */ 
    private Class formFieldType;
    
    
    //********************************* Constructors and initialization methods

    /**
     * Construct an instance with no fields specified
     */
    public PropertyMapping() {
    }

    /**
     * <p>Construct an instance specifying the basic details about a field.</p>
     * 
     * @param name the name of the property
     * @param type the type of the property
     * @param getter the getter method in the bean type
     * @param setter the setter method in the bean type
     * 
     * @deprecated Use the constructor that specifies the formFieldType.
     */
    public PropertyMapping(String name, Class type, Method getter, Method setter) {
        this.name = name;
        this.type = type;
        this.getter = getter;
        this.setter = setter;
    }

    /**
     * <p>Construct an instance specifying the basic details about a field.</p>
     * 
     * @param name  the name of the property
     * @param type  the type of the property
     * @param getter    the getter method in the bean type
     * @param setter    the setter method in the bean type
     * @param formFieldType     the type of this property on the generated form
     */
    public PropertyMapping(String name, 
                           Class type, Method getter, Method setter,
                           Class formFieldType) {
        this.name = name;
        this.type = type;
        this.getter = getter;
        this.setter = setter;
        this.formFieldType = formFieldType;
    }

    /**
     * Set the converter object and a possible conversion parameter
     * @param converter the object used to convert from String to this property's type
     * @param conversionParam conversion parameter, when necessary
     */
    public void setConverter(Object converter,
                             String conversionParam) {
        this.converter = converter;
        this.conversionParam = conversionParam;
        this.conversionKey = null;
        this.conversionBundle = null;
        
        if (converter instanceof FormConverter) {
            if (conversionParam == null) {
                conversionParam = formName;
            }
        }
    }

    /**
     * Set the converter object and the resource bundle and key to use as conversion parameter
     * @param converter the object used to convert from String to this property's type
     * @param conversionKey the message resource key for the String to use as conversion parameter
     * @param conversionBundle the bundle key of the resource file 
     *  from which to load the conversion parameter
     */
    public void setConverter(Object converter,
                             String conversionKey,
                             String conversionBundle) {
        this.converter = converter;
        this.conversionParam = null;
        this.conversionKey = conversionKey;
        this.conversionBundle = conversionBundle;
        
        if (converter instanceof FormConverter) {
            if (conversionParam == null) {
                conversionParam = formName;
            }
        }
    }
    
    //******************************************************** Accessor methods

    /** 
     * The name of the property on the form bean. 
     */
    public String getName() {
        return name;
    }

    /**
     * The class of the property on the business object.
     */ 
    public Class getType() {
        return type;
    }

    /**
     * The name of the form bean that will hold the data
     * for this property.
     */ 
    public String getFormName() {
        return formName;
    }

    /**
     * Set the name of the form bean that holds the data
     * for this property.
     */ 
    public void setFormName(String formName) {
        this.formName = formName;
    }

    /**
     * The getter method of the business object to use for this property.
     */ 
    public Method getGetter() {
        return getter;
    }

    /**
     * The setter method of the business object to use for this property.
     */ 
    public Method getSetter() {
        return setter;
    }

    /**
     * The converter object to use for this property.
     */ 
    public Object getConverter() {
        return converter;
    }

    /**
     * The conversion parameter to use when converting this property.
     */ 
    public String getConversionParam() {
        return conversionParam;
    }

    /**
     * The resource key that points to the conversion parameter for this 
     * property.
     */ 
    public String getConversionKey() {
        return conversionKey;
    }

    /**
     * The name of the resource bundle where the conversion parameter is 
     * located.
     */ 
    public String getConversionBundle() {
        return conversionBundle;
    }

    /**
     * The default value to use for this property.
     */ 
    public String getDefaultValue() {
        return defaultValue;
    }

    /**
     * Set the default value to use for this property.
     */ 
    public void setDefaultValue(String defaultValue) {
        this.defaultValue = defaultValue;
    }

    /**
     * Get the object holding arbitrary properties for this form property.
     * These properties are passed onto the converter. 
     */ 
    public Properties getProperties() {
        return properties;
    }
    
    /**
     * Set the object holding arbitrary properties for this form property. 
     * These properties are passed onto the converter. 
     */ 
    public void setProperties(Properties properties) {
        this.properties = properties;
    }

    /**
     * The Class used for this field in the form bean.
     */
    public Class getFormFieldType() {
        return formFieldType;
    }
    
    
    //************************************************************** toString()

    
    /**
     * Returns a String representation of this class for logging and
     * debugging purposes.
     */ 
    public String toString() {
        StringBuffer result = new StringBuffer(256);
        result.append("\n\tPropertyMapping [");
        result.append("\n\t\tname=              ").append(getName()).append(";");
        result.append("\n\t\ttype=              ").append(getType()).append(";");
        if (getFormName() != null) {
            result.append("\n\t\tformName=          ").append(getFormName()).append(";");
        }
        result.append("\n\t\tgetter=            ").append(getGetter()).append(";");
        result.append("\n\t\tsetter=            ").append(getSetter()).append(";");
        result.append("\n\t\tconverter=         ").append(getConverter()).append(";");
        if (getConversionParam() != null) {
            result.append("\n\t\tconversionParam=   ").append(getConversionParam()).append(";");
        }
        if (getConversionKey() != null) {
            result.append("\n\t\tconversionKey=     ").append(getConversionKey()).append(";");
        }
        if (getConversionBundle() != null) {
            result.append("\n\t\tconversionBundle=  ").append(getConversionBundle()).append(";");
        }
        if (getDefaultValue() != null) {
            result.append("\n\t\tdefaultValue=      ").append(getDefaultValue()).append(";");
        }
        result.append("\n\t]");
        return result.toString();
    }

}
