package formdef.plugin.config;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.beanutils.PropertyUtils;

import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Properties;
import java.beans.PropertyDescriptor;

import formdef.plugin.conversion.Converter;
import formdef.plugin.FormMapping;
import formdef.plugin.PropertyMapping;

/**
 * <p>Holds the configuration information for a single form.</p>
 * @author Hubert Rabago
 */
public class FormMappingConfig {

    private static final Log log = LogFactory.getLog(FormMappingConfig.class);

    /** The given name of the form. */
    protected String name;

    /**
     * The fully qualified name of the class used
     * to define the fields of this form.
     */
    protected String beanType;

    /**
     * The fully qualified name of the ActionForm class
     * to be used for this form.
     */
    protected String formType;
    
    /**
     * The fully qualified name of the FormBeanConfig class
     * to be used for this form.
     */ 
    protected String configType; 

    /** The name of the class to be used as factory for this form's bean. */
    protected String factory;

    /**
     * The name of the method in the factory which generates this form's bean.
     */
    protected String factoryMethodName;

    /**
     * True if the name of the form should be passed to the factory
     * method when called to generate a bean.
     */
    protected String factoryMethodNameParam = "false";

    /** A list of properties for this form */
    protected List properties;
    
    /** 
     * Arbitrary properties to be passed to the FormBeanConfig instance
     * of this form.
     */
    protected Properties configProperties; 
    
    
    /** 
     * The comma-separated list of properties for which a form field will
     * not be created. 
     */
    protected String excludes;
    
    
    /**
     * The list of excluded properties as configured through {@link #excludes}.
     */ 
    protected List excludedProperties;


    public FormMappingConfig() {
        properties = new ArrayList();
        configProperties = new Properties();
    }

    /**
     * Create an instance with the name and beanType specified.
     * @param name the given name of this form instance
     * @param beanType the class name used to defined this form
     */
    public FormMappingConfig(String name, String beanType) {
        this();
        this.name = name;
        this.beanType = beanType;
    }
    
    //*************************************************************************
    // Accessors

    /** The given name of the form. */
    public String getName() {
        return name;
    }

    /** The given name of the form. */
    public void setName(String name) {
        this.name = name;
    }

    /**
     * The fully qualified name of the class used
     * to define the fields of this form.
     */
    public String getBeanType() {
        return beanType;
    }

    /**
     * Set the name of the bean type associated with this config's form.
     * If the given String is empty, beanType is set to null.
     * @param beanType the fully qualified name of a Java object
     */
    public void setBeanType(String beanType) {
        this.beanType = beanType;
        if (beanType != null && beanType.length() <= 0) {
            this.beanType = null;
        }
    }

    /**
     * The fully qualified name of the ActionForm class
     * to be used for this form.
     */
    public String getFormType() {
        return formType;
    }

    /**
     * The fully qualified name of the ActionForm class
     * to be used for this form.
     */
    public void setFormType(String formType) {
        this.formType = formType;
    }

    public String getConfigType() {
        return configType;
    }

    public void setConfigType(String configType) {
        this.configType = configType;
    }

    /** The name of the class to be used as factory for this form's bean. */
    public String getFactory() {
        return factory;
    }

    /** The name of the class to be used as factory for this form's bean. */
    public void setFactory(String factory) {
        this.factory = factory;
    }

    /**
     * The name of the method in the factory which generates this form's bean.
     */
    public String getFactoryMethodName() {
        return factoryMethodName;
    }

    /**
     * The name of the method in the factory which generates this form's bean.
     */
    public void setFactoryMethodName(String factoryMethodName) {
        this.factoryMethodName = factoryMethodName;
    }

    /**
     * "true" if the name of the form should be passed to the factory
     * method when called to generate a bean.
     */
    public String getFactoryMethodNameParam() {
        return factoryMethodNameParam;
    }

    /**
     * "true" if the name of the form should be passed to the factory
     * method when called to generate a bean.
     */
    public void setFactoryMethodNameParam(String factoryMethodNameParam) {
        this.factoryMethodNameParam = factoryMethodNameParam;
    }

    /** 
     * The comma-separated list of properties for which a form field will
     * not be created. 
     */
    public String getExcludes() {
        return excludes;
    }

    /** 
     * The comma-separated list of properties for which a form field will
     * not be created. 
     */
    public void setExcludes(String excludes) {
        this.excludes = excludes;
    }

    /**
     * Add configuration information for a property of this form
     * @param propertyConfig an instance of {@link formdef.plugin.config.PropertyMappingConfig}
     * containing info on a property being added
     */
    public void addProperty(PropertyMappingConfig propertyConfig) {
        properties.add(propertyConfig);
    }

    /**
     * Get the {@link formdef.plugin.config.PropertyMappingConfig} with the given name
     * @param propertyName the name of the property to return
     * @return the property with the given name
     */
    public PropertyMappingConfig getProperty(String propertyName) {
        if (propertyName == null) {
            return null;
        }
        for (int i = 0; i < properties.size(); i++) {
            PropertyMappingConfig config = (PropertyMappingConfig) properties.get(i);
            if ((config != null) && (propertyName.equals(config.getName()))) {
                return config;
            }
        }
        return null;
    }

    /**
     * Arbitrary properties to be passed to the FormBeanConfig instance
     * of this form.
     */ 
    public Properties getConfigProperties() {
        return configProperties;
    }

    /**
     * Arbitrary properties to be passed to the FormBeanConfig instance
     * of this form.
     */ 
    public void setConfigProperties(Properties configProperties) {
        this.configProperties = configProperties;
    }

    /**
     * <p>Add an arbitrary property to be passed on to the FormBeanConfig
     * instance that will be used to configure this form in Struts.</p>
     * 
     * @param key   the key for the arbitrary property.
     * @param value the value of the property.
     */ 
    public void addConfigProperty(String key, String value) {
        configProperties.setProperty(key, value);
    }
    
    
    //*************************************************************************
    // Form-generation methods
    
    
    /**
     * Generate a {@link formdef.plugin.FormMapping} object to hold the information
     *      about this instance
     * @throws java.lang.ClassNotFoundException if the type for this
     *          form (or its specified property) cannot be found
     * @throws java.lang.NoSuchMethodException if the specified factory
     *          method cannot be found
     * @throws InvocationTargetException from a constructor call
     * @throws IllegalAccessException from a constructor call
     * @throws InstantiationException from a constructor call
     */ 
    public FormMapping generateForm(FormDefConfig formDefConfig)
            throws Exception {

        // declare the variable that will hold this method's result
        FormMapping form = null;
        Class associatedType = null;

        // is there an associated bean type?
        if (getBeanType() != null) {
            // get the Class of this form's beanType
            associatedType = Class.forName(getBeanType());
            form = new FormMapping(getName(),associatedType);

            // get the factory Method to use for this beanType
            if ((factory != null) && (factoryMethodName != null)) {
                initializeFactory(form);

            }

            // resolve add'l attributes for this form's properties
            resolveProperties(associatedType);
        } else {
            // create a form def that's not associated with any type
            form = new FormMapping(getName(),null);
        }

        // create the properties for this form
        Map formProperties = form.getProperties();
        for (int i = 0; i < this.properties.size(); i++) {
            PropertyMappingConfig config = (PropertyMappingConfig) this.properties.get(i);
            if (isExcluded(config)) {
                if (log.isTraceEnabled()) {
                    log.trace("Excluding property: " + config.getName());
                }
                continue;
            }
            PropertyMapping property = config.generateProperty(form, formDefConfig);
            applyGlobalConverters(property,config,formDefConfig);
            if (log.isTraceEnabled()) {
                log.trace("Created property: " + property);
            }
            formProperties.put(config.getName(),property);
        }

        return form;
    }

    
    /**
     * <p>Returns true if the property that the given config object represents
     * should not be included in the form that's being generated.</p>
     * 
     * @param config    the configuration of the property being inspected.
     * 
     * @return true if this property should not be present on the form.
     */ 
    private boolean isExcluded(PropertyMappingConfig config) {
        if (config == null) {
            return false;
        }
        
        // check if this property was specifically configured to be excluded
        if (config.getExclude() != null) {
            if (Boolean.valueOf(config.getExclude()).booleanValue()) {
                return true;
            }
        }
        
        // check if this property is included in our comma-seperated list of 
        //      excluded fields
        String propertyName = config.getName();
        if (getExcludedProperties().contains(propertyName)) {
            return true;
        }
        
        // if we reach this point, then this property should be included
        return false;
    }
    
    
    /**
     * <p>Get the populated list of property names that were configured to
     * be excluded from the generated form through the {@link #excludes}
     * attribute.</p>
     * 
     * @return a list of names of properties to be excluded.
     */ 
    protected List getExcludedProperties() {
        // return the list if we already have it
        if (excludedProperties != null) {
            return excludedProperties;
        }
        
        // but since we don't, let's create it now
        excludedProperties = new ArrayList();
        
        if (excludes == null) {
            return excludedProperties;
        }
        
        // parse the csv list of excluded fields
        StringTokenizer st = new StringTokenizer(excludes, ", \t\n\r\f");
        while (st.hasMoreTokens()) {
            excludedProperties.add(st.nextToken());
        }
        
        return excludedProperties;
    }
    

    /**
     * Initialize the form's designated factory
     * @param form the form being initialized
     * @throws Exception if an error occurs while creating the factory object
     * or locating the factory method
     */ 
    protected void initializeFactory(FormMapping form) 
            throws Exception {
        Class factoryType = Class.forName(factory);
                
        // instantiate the factory object using a no-arg constructor
        Constructor constructor = 
                factoryType.getConstructor(new Class[] {});
        Object factory = constructor.newInstance(new Object[] {});
                
        // find the factory method
        Method factoryMethod = null;
        boolean useMethodNameParameter =
                "true".equals(factoryMethodNameParam);
        if (useMethodNameParameter) {
            factoryMethod = factoryType.getMethod(
                    factoryMethodName, new Class[] {String.class});
        } else {
            factoryMethod = factoryType.getMethod(
                    factoryMethodName, new Class[] {});
        }
                
        // pass them all to the form
        form.setFactoryMethod(factory, 
                factoryMethod,
                useMethodNameParameter);
    }

    /**
     * Find a global converter that applies to the given property
     * @param property the property being processed
     * @param config the configuration of the property being processed
     * @param formDefConfig the configuration information
     *      applicable to the current form
     */
    private void applyGlobalConverters(PropertyMapping property,
                                       PropertyMappingConfig config,
                                       FormDefConfig formDefConfig) {

        // Find a global converter that's applicable for this property
        // First, check if there's a named converter to use
        String name = config.getConverterName();
        if ((name != null) && (name.length() > 0)) {
            GlobalConverterConfig converterConfig =
                    formDefConfig.findConverterNameConverter(name);
            if (converterConfig != null) {
                applyConverterConfiguration(property, converterConfig);
                return;
            }
        }

        // Second, check if this property's name
        //      has a converter configured for it
        GlobalConverterConfig converterConfig =
                formDefConfig.findPropertyNameConverter(property.getName());
        if (converterConfig != null) {
            applyConverterConfiguration(property, converterConfig);
            return;
        }

        // Lastly, check if this property's type
        //      has a converter configured for it
        converterConfig =
                formDefConfig.findPropertyTypeConverter(
                        property.getType());
        log.trace("converterConfig=" + converterConfig);
        if (converterConfig != null) {
            log.trace("applying to " + property);
            applyConverterConfiguration(property, converterConfig);
            log.trace("property=" + property);
            return;
        }
    }

    /**
     * Applies the converter configuration to the converter of the property
     * @param property the property to modify
     * @param converterConfig the configuration to apply
     */
    private void applyConverterConfiguration(
            PropertyMapping property,
            GlobalConverterConfig converterConfig) {
        Converter converter = (Converter) property.getConverter();
        String param = property.getConversionParam();
        String paramKey = property.getConversionKey();
        String paramBundle = property.getConversionBundle();
        if (converterConfig.getTypeName() != null) {
            converter = (Converter) converterConfig.getConverter();
        }

        // apply the global values where there are no property values 
        if (converterConfig.getConversionParameter() != null) {

            // only use the global conv param if there wasn't anything
            //  specified specially for this property
            if (param == null) {
                param = converterConfig.getConversionParameter();
            } else {
                if (log.isTraceEnabled()) {
                    log.trace(property.getName() + " converter: "
                            + " A conversion parameter was specified for this;"
                            + " this will take precedence over"
                            + " the global setting.");
                }
            }
        }

        if (converterConfig.getConversionKey() != null) {

            // only use the global conv param if there wasn't anything
            //  specified specially for this property
            if (paramKey == null) {
                paramKey = converterConfig.getConversionKey();
                paramBundle = converterConfig.getConversionBundle();
            } else {
                if (log.isTraceEnabled()) {
                    log.trace(property.getName() + " converter: "
                            + " A conversion parameter was specified for this;"
                            + " this will take precedence over"
                            + " the global setting.");
                }
            }
        }

        //if (log.isTraceEnabled()) {
        //    log.trace(property.getName() + " converter:"
        //          + " Using [" + converter + "," + param + ".");
        //}
        if (param != null) {
            property.setConverter(converter, param);
        } else {
            property.setConverter(converter, paramKey, paramBundle);
        }
    }


    /**
     * Inspect the current attributes of the properties of this form and resolve
     * missing setter and getter names.
     */
    protected void resolveProperties(Class associatedType) {

        // get the PropertyDescriptors for this beanType's properties
        PropertyDescriptor descriptors[] =
                PropertyUtils.getPropertyDescriptors(associatedType);

        // assign the getters and setters for each property
        for (int i = 0; i < this.properties.size(); i++) {
            PropertyMappingConfig config = 
                    (PropertyMappingConfig) this.properties.get(i);

            PropertyDescriptor descriptor = null;
            for (int j = 0; j < descriptors.length; j++) {
                PropertyDescriptor temp = descriptors[j];
                if (temp.getName().equals(config.getName())) {
                    descriptor = temp;
                    break;
                }
            }

            if (config.getGetter() == null) {
                // no getter was specified.  see if we can assign one.
                if (descriptor != null) {
                    Method getter = PropertyUtils.getReadMethod(descriptor);
                    config.setGetter(getter.getName());
                }
            }

            if (config.getSetter() == null) {
                // no setter was specified.  see if we can assign one.
                if (descriptor != null) {
                    Method setter = PropertyUtils.getWriteMethod(descriptor);
                    config.setSetter(setter.getName());
                }
            }

        }

        // get the bean properties that we don't have yet
        for (int j = 0; j < descriptors.length; j++) {
            PropertyDescriptor descriptor = descriptors[j];
            
            // first, verify that this isn't an excluded property
            if (getExcludedProperties().contains(descriptor.getName())) {
                // ack! it IS excluded.  skip it then.
                continue;
            }

            // if this prop has a getter and setter,
            //      we'll make sure it's in the list
            Method getter = PropertyUtils.getReadMethod(descriptor);
            Method setter = PropertyUtils.getWriteMethod(descriptor);
            if ((getter != null) && (setter != null)) {
                // is this already on the list?
                boolean isOnTheList = false;
                for (int i = 0; i < this.properties.size(); i++) {
                    PropertyMappingConfig config =
                            (PropertyMappingConfig) this.properties.get(i);
                    if (descriptor.getName().equals(config.getName())) {
                        isOnTheList = true;
                        break;
                    }
                }
                if (isOnTheList) {
                    // it's already on the list,
                    //      so let's look for another property
                    continue;
                }

                PropertyMappingConfig config = new PropertyMappingConfig();
                config.setName(descriptor.getName());
                config.setSetter(setter.getName());
                config.setGetter(getter.getName());
                properties.add(config);
            }
        }

    }


    /**
     * Returns a String representation of this object.
     */
    public String toString() {
        StringBuffer result = new StringBuffer(256);
        result.append("FormMappingConfig [");
        result.append("name=").append(getName()).append(";");
        if (beanType != null) {
            result.append("beanType=").append(beanType).append(";");
        }
        if (formType != null) {
            result.append("formType=").append(formType).append(";");
        }
        if (factory != null) {
            result.append("factory=").append(factory).append(";");
        }
        if (factoryMethodName != null) {
            result.append("factoryMethodName=").append(
                    factoryMethodName).append(";");
        }
        if (factoryMethodNameParam != null) {
            result.append("factoryMethodNameParam=").append(
                    getFactoryMethodNameParam()).append(";");
        }
        if (excludes != null) {
            result.append("excludes=").append(excludes).append(";");
        }
        for (int i = 0; i < properties.size(); i++) {
            PropertyMappingConfig config = (PropertyMappingConfig) properties.get(i);
            result.append("\n").append(config);
        }
        result.append("\n]");
        return result.toString();
    }

}
