package formdef.plugin.config;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.config.FormPropertyConfig;

import java.lang.reflect.Method;
import java.lang.reflect.Constructor;
import java.util.Properties;
import java.util.Enumeration;

import formdef.plugin.conversion.ConverterFactory;
import formdef.plugin.PropertyMapping;
import formdef.plugin.FormMapping;


/**
 * Holds the configuration specs provided for a bean property as read from the
 * FormDef configuration file.
 * <p/>
 * 
 * @author Hubert Rabago
 */
public class PropertyMappingConfig {

    private static final Log log = LogFactory.getLog(PropertyMappingConfig.class);

    /** 
     * The name of the property on the form bean. 
     */
    private String name;
    
    /**
     * The name of the getter method in the associated java 
     *   bean that will be used to retrieve a value when 
     *   populating this field in the form bean.
     */ 
    private String getter;
    
    /**
     * The name of the setter method in the associated java
     * bean that will be used to pass the value from 
     * the form bean.
     */ 
    private String setter;
    
    /**
     * The conversion parameter to pass to the converter for this field.
     */ 
    private String conversionParam;
    
    /**
     * The key in a resource property file which specifies
     * the conversion parameter to be used with this field.
     */ 
    private String conversionKey;
    
    /**
     * The resource property bundle name from which to get
     * the conversion parameter; used with the "key" attribute.
     */ 
    private String conversionBundle;
    
    /**
     * The name of the converter to use for this field, as
     * provided by a global converter defined for a
     * "converter-name".
     */ 
    private String converterName;
    
    /**
     * The fully qualified class name of the Converter
     * implementation to use for this field.
     */ 
    private String converterType;

    /**
     * Equivalent to the "initial" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */ 
    private String initial;
    
    /** 
     * Equivalent to the "reset" attribute of the "form-property" element
     * when defining a form bean in the struts configuration file.
     * This property isn't used in Struts versions earlier than 1.3.
     */ 
    private String reset;
    
    /**
     * Equivalent to the "size" attribute of the 
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */ 
    private String size;
    
    /**
     * The fully qualified class name to use for this field
     * when defining the form bean.
     * Equivalent to the "type" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */ 
    private String type;
    
    /**
     * The name of the form bean that will hold the data
     * for this property.
     */ 
    private String formName;
    
    /**
     * "true" if this field should not be added to the form that will be
     * generated.
     */
    private String exclude;
    
    /**
     * Arbitrary properties configured for this property's converter.   
     */ 
    private Properties converterProperties = new Properties();

    /**
     * The fully qualified class name of the FormPropertyConfig class
     * to use when configuring this property in Struts.
     */ 
    private String configType;
    
    /**
     * Arbitrary properties for the FormPropertyConfig instance
     * used to configure this property in Struts.
     */ 
    private Properties configProperties = new Properties();

    
    /**
     * Create an instance of this object and initialize internal properties
     */
    public PropertyMappingConfig() {
    }
    
    
    //*************************************************************** Accessors

    /** 
     * The name of the property on the form bean. 
     */
    public String getName() {
        return name;
    }

    /** 
     * The name of the property on the form bean. 
     */
    public void setName(String name) {
        this.name = name;
    }
    
    /**
     * The name of the getter method in the associated java 
     *   bean that will be used to retrieve a value when 
     *   populating this field in the form bean.
     */ 
    public String getGetter() {
        return getter;
    }

    /**
     * The name of the getter method in the associated java 
     *   bean that will be used to retrieve a value when 
     *   populating this field in the form bean.
     */
    public void setGetter(String getter) {
        this.getter = getter;
    }

    /**
     * The name of the setter method in the associated java
     * bean that will be used to pass the value from 
     * the form bean.
     */
    public String getSetter() {
        return setter;
    }

    /**
     * The name of the setter method in the associated java
     * bean that will be used to pass the value from 
     * the form bean.
     */
    public void setSetter(String setter) {
        this.setter = setter;
    }

    /**
     * The conversion parameter to pass to the converter for this field.
     */
    public String getConversionParam() {
        return conversionParam;
    }

    /**
     * The conversion parameter to pass to the converter for this field.
     */
    public void setConversionParam(String conversionParam) {
        this.conversionParam = conversionParam;
    }

    /**
     * The key in a resource property file which specifies
     * the conversion parameter to be used with this field.
     */
    public String getConversionKey() {
        return conversionKey;
    }

    /**
     * The key in a resource property file which specifies
     * the conversion parameter to be used with this field.
     */
    public void setConversionKey(String conversionKey) {
        this.conversionKey = conversionKey;
    }

    /**
     * The resource property bundle name from which to get
     * the conversion parameter; used with the "key" attribute.
     */
    public String getConversionBundle() {
        return conversionBundle;
    }

    /**
     * The resource property bundle name from which to get
     * the conversion parameter; used with the "key" attribute.
     */
    public void setConversionBundle(String conversionBundle) {
        this.conversionBundle = conversionBundle;
    }

    /**
     * The fully qualified class name of the Converter
     * implementation to use for this field.
     */
    public String getConverterType() {
        return converterType;
    }

    /**
     * The fully qualified class name of the Converter
     * implementation to use for this field.
     */
    public void setConverterType(String converterType) {
        this.converterType = converterType;
    }

    /**
     * The name of the converter to use for this field, as
     * provided by a global converter defined for a
     * "converter-name".
     */
    public String getConverterName() {
        return converterName;
    }

    /**
     * The name of the converter to use for this field, as
     * provided by a global converter defined for a
     * "converter-name".
     */
    public void setConverterName(String converterName) {
        this.converterName = converterName;
    }

    /**
     * Equivalent to the "initial" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public String getInitial() {
        return initial;
    }

    /**
     * Equivalent to the "initial" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public void setInitial(String initial) {
        this.initial = initial;
    }

    /** 
     * Equivalent to the "reset" attribute of the "form-property" element
     * when defining a form bean in the struts configuration file.
     * This property isn't used in Struts versions earlier than 1.3.
     */
    public String getReset() {
        return reset;
    }

    /** 
     * Equivalent to the "reset" attribute of the "form-property" element
     * when defining a form bean in the struts configuration file.
     * This property isn't used in Struts versions earlier than 1.3.
     */
    public void setReset(String reset) {
        this.reset = reset;
    }

    /**
     * Equivalent to the "size" attribute of the 
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public String getSize() {
        return size;
    }

    /**
     * Equivalent to the "size" attribute of the 
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public void setSize(String size) {
        this.size = size;
    }

    /**
     * The fully qualified class name to use for this field
     * when defining the form bean.
     * Equivalent to the "type" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public String getType() {
        return type;
    }

    /**
     * The fully qualified class name to use for this field
     * when defining the form bean.
     * Equivalent to the "type" attribute of the
     * "form-property" element when defining a form bean
     * in the struts configuration file.
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * The name of the form bean that will hold the data
     * for this property.
     */
    public String getFormName() {
        return formName;
    }

    /**
     * The name of the form bean that will hold the data
     * for this property.
     */
    public void setFormName(String formName) {
        this.formName = formName;
    }

    /**
     * "true" if this field should not be added to the form that will be
     * generated.
     */
    public String getExclude() {
        return exclude;
    }

    /**
     * "true" if this field should not be added to the form that will be
     * generated.
     */
    public void setExclude(String exclude) {
        this.exclude = exclude;
    }

    /**
     * @deprecated Use {@link #getConverterProperties()} instead.  This
     * method will be removed in a future release.
     */ 
    public Properties getProperties() {
        return getConverterProperties();
    }

    /**
     * @deprecated Use {@link #setConverterProperties} instead.  This
     * method will be removed in a future release.
     */ 
    public void setProperties (Properties converterProperties) {
        setConverterProperties(converterProperties);
    }

    /**
     * Arbitrary properties configured for this form property's Converter.   
     */
    public Properties getConverterProperties() {
        return converterProperties;
    }

    /**
     * Arbitrary properties configured for this form property's Converter.   
     */
    public void setConverterProperties(Properties converterProperties) {
        this.converterProperties = converterProperties;
    }
    
    /**
     * Add an arbitrary property for this form property's Converter.   
     */
    public void addConverterProperty(String key, String value) {
        converterProperties.setProperty(key, value);
    }

    public String getConfigType() {
        return configType;
    }

    public void setConfigType(String configType) {
        this.configType = configType;
    }

    /**
     * Arbitrary properties configured for this property's FormPropertyConfig 
     * instance.
     */
    public Properties getConfigProperties() {
        return configProperties;
    }

    /**
     * Arbitrary properties configured for this property's FormPropertyConfig 
     * instance.   
     */
    public void setConfigProperties(Properties configProperties) {
        this.configProperties = configProperties;
    }
    
    /**
     * Add an arbitrary property for this property's FormPropertyConfig 
     * instance. 
     */
    public void addConfigProperty(String key, String value) {
        configProperties.setProperty(key, value);
    }

    /**
     * Return a Class corresponds to the value specified for the
     * <code>type</code> property.
     * @deprecated use the {@link #getTypeClass(FormDefConfig)} method instead.
     */
    public Class getTypeClass() {
        if ((getType() == null) || (getType().length() <= 0)) {
            return String.class;
        }
        // utilize the method in FormPropertyConfig
        FormPropertyConfig formPropertyConfig = new FormPropertyConfig();
        formPropertyConfig.setType(getType());
        return formPropertyConfig.getTypeClass();
    }

    /**
     * Return a Class corresponds to the value specified for the
     * <code>type</code> property.
     */
    protected Class getTypeClass(FormDefConfig formDefConfig) {
        String localType = getType();
        
        // if there was no type provided, check if there's a form specified
        if ((localType == null) || (localType.length() <= 0)) {
            if (getFormName() != null) {
                localType = formDefConfig.getFormType(getFormName());
            }
        }
        
        // if we still don't have a type, use String
        if ((localType == null) || (localType.length() <= 0)) {
            return String.class;
        }
        
        // utilize the method in FormPropertyConfig
        FormPropertyConfig formPropertyConfig = new FormPropertyConfig();
        formPropertyConfig.setType(localType);
        return formPropertyConfig.getTypeClass();
    }

    /**
     * <p>Generate a {@link PropertyMapping} object from the 
     *      information held by this object.</p>
     * 
     * @param form          the {@link FormMapping} to which this property 
     *                          belongs.
     * @param formDefConfig the FormDefConfig instance under which the form
     *                          is configured.
     * @return a {@link formdef.plugin.PropertyMapping PropertyMapping} 
     *      instance based on this object
     */
    public PropertyMapping generateProperty(FormMapping form,
                                            FormDefConfig formDefConfig)
            throws ClassNotFoundException, NoSuchMethodException {

        Class formType = form.getBeanType();

        // find the getter method
        Method getterMethod = null;
        if ((getGetter() != null) && (getGetter().length() > 0)) {
            try {
                getterMethod = formType.getMethod(getGetter(),new Class[]{});
            } catch (Exception e) {
                getterMethod = null;
            }
        }

        // look for the type of this property by checking the result of the
        //  getter method, or use a String type as default
        Class propertyType =
                (getterMethod!=null)?getterMethod.getReturnType():String.class;

        // find the setter method
        Method setterMethod = null;
        if ((getSetter() != null) && (getSetter().length() > 0)) {
            try {
                setterMethod = formType.getMethod(
                        getSetter(),new Class[]{propertyType});
            } catch (Exception e) {
                setterMethod = null;
            }
        }

        // create the property instance
        PropertyMapping property =
                new PropertyMapping(getName(),
                        propertyType, getterMethod, setterMethod,
                        getTypeClass(formDefConfig));

        // pass the name of the form this property is associated with, if any;
        //  this value must be set before property.setConverter() is called.
        if (formName != null) {
            property.setFormName(formName);
        }

        // get a reference to the converter that we
        //      will be using for this property
        Object converterInstance = null;
        if (converterType != null) {
            // a converterType was specified for this property;
            //      let's instantiate it
            Class converterClass = Class.forName(converterType);
            Constructor constructor =
                    converterClass.getConstructor(new Class[]{});

            try {
                converterInstance = constructor.newInstance(new Object[]{});
            } catch (Exception e) {
                log.error("Exception [" + e + "," + e.getMessage() + "]", e);
                throw new RuntimeException(e);
            }

        } else {
            // use the standard converter for this type
            Class typeClass = getTypeClass(formDefConfig);
            converterInstance = 
                    ConverterFactory.getInstance().getConverter(
                            propertyType, typeClass, this);
        }

        // pass the converter we got to the property we're building
        if (conversionParam != null) {
            property.setConverter(converterInstance,conversionParam);
        } else {
            property.setConverter(converterInstance,conversionKey,conversionBundle);
        }
        
        // copy the arbitrary properties from this config object
        Properties newProperties = new Properties();
        Enumeration keys = converterProperties.keys();
        while (keys.hasMoreElements()) {
            String key = (String) keys.nextElement();
            newProperties.setProperty(key, converterProperties.getProperty(key));
        }
        property.setProperties(newProperties);
        
        return property;
    }


    public String toString() {
        StringBuffer result = new StringBuffer(256);
        result.append("PropertyMappingConfig [");
        result.append("name=").append(name).append(";");
        if (type != null) {
            result.append("type=").append(type).append(";");
        }
        if (getter != null) {
            result.append("getter=").append(getter).append(";");
        }
        if (setter != null) {
            result.append("setter=").append(setter).append(";");
        }
        if (converterType != null) {
            result.append("converterType=").append(converterType).append(";");
        }
        if (converterName != null) {
            result.append("converterName=").append(converterName).append(";");
        }
        if (conversionParam != null) {
            result.append("conversionParam=").append(conversionParam).append(";");
        }
        if (conversionKey != null) {
            result.append("conversionKey=").append(conversionKey).append(";");
        }
        if (conversionBundle != null) {
            result.append("conversionBundle=").append(conversionBundle).append(";");
        }
        if (initial != null) {
            result.append("initial=").append(initial).append(";");
        }
        if (reset != null) {
            result.append("reset=").append(reset).append(";");
        }
        if (size != null) {
            result.append("size=").append(size).append(";");
        }
        if (type != null) {
            result.append("type=").append(type).append(";");
        }
        if (formName != null) {
            result.append("formName=").append(formName).append(";");
        }
        if (exclude != null) {
            result.append("exclude=").append(exclude).append(";");
        }
        
        result.append("]");
        return result.toString();
    }

}
