package formdef.plugin.conversion;

import org.apache.struts.action.ActionForm;
import org.apache.struts.util.RequestUtils;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;
import formdef.plugin.config.PropertyMappingConfig;

/**
 * Factory for standard converter objects.
 * 
 * @author Hubert Rabago
 */
public class ConverterFactory {

    private static Object NUMBER_CONVERTER;
    private static Object BOOLEAN_CONVERTER;
    private static Object CHAR_CONVERTER;
    private static Object STRING_CONVERTER;
    private static Object DATE_CONVERTER;
    private static Object FORM_CONVERTER;
    
    
    /** Logging utility class instance. */
    protected static Log log = LogFactory.getLog(ConverterFactory.class);


    /**
     * The ConverterFactory instance used by the application.
     */ 
    protected static ConverterFactory instance = null;


    /**
     * The classname of the ConverterFactory instance that will be used
     * by the application.
     */ 
    protected static String converterFactoryClassName = 
            ConverterFactory.class.getName();

    
    //*************************************************************************
    // Static methods
    

    /**
     * Return the ConverterFactory instance configured for this module.
     * @return the ConverterFactory to use
     */ 
    public static ConverterFactory getInstance() {
        if (instance == null) {
            try {
                // use RequestUtils to instantiate the ConverterFactory class
                instance = (ConverterFactory) RequestUtils
                        .applicationInstance(converterFactoryClassName);
            } catch (Exception e) {
                log.error("Exception [" + e + "," + e.getMessage() + "]", e);

                String message =
                        "Unable to create ConverterFactory object: "
                        + converterFactoryClassName;
                throw new IllegalArgumentException(message);
            }
        }
        return instance;
    }

    
    /**
     * Set the class name of the ConverterFactory instance that will be used
     * in the application.
     */ 
    public static void setConverterFactoryClassName(String name) {
        ConverterFactory.converterFactoryClassName = name;
    }
    
    
    //*************************************************************************
    // Public API methods
    

    /**
     * <p>Return an object for use in converting the specified beanFieldType
     * to the formFieldType and vice versa.</p>
     * 
     * @param beanFieldType the {@link Class} for the field on the bean.
     * @param formFieldType the {@link Class} for the field on the form.
     * @param propertyMappingConfig configuration information for the property
     *                      that will use the converter.
     * @return an object suitable for use as a converter for the property
     *          described by the given propertyMappingConfig.
     */
    public Object getConverter(Class beanFieldType,
                               Class formFieldType,
                               PropertyMappingConfig propertyMappingConfig) {
        
        Object result = null;

        if (formFieldType.equals(String.class)) {
            
            result = getStringConverter(beanFieldType);
            
        } else {
            // check for other form types supported
            
            if (propertyMappingConfig.getFormName() != null) { 
                if (ActionForm.class.isAssignableFrom(formFieldType)) {
                    if (FORM_CONVERTER == null) {
                        FORM_CONVERTER = new FormConverter();
                    }
                    result = FORM_CONVERTER;
                }
            }
            
        }

        return result;
    }


    /**
     * <p>Return an object for use in converting the specified beanFieldType
     * to the formFieldType and vice versa.</p>
     * 
     * @param beanFieldType the {@link Class} for the field on the bean
     * @param formFieldType the {@link Class} for the field on the form
     * @return an object suitable for use as a converter between the  
     * beanFieldType and formFieldType.
     */
    public Object getConverter(Class beanFieldType, Class formFieldType) {
        
        Object result = null;

        if (formFieldType.equals(String.class)) {
            
            result = getStringConverter(beanFieldType);
            
        } else {
            // check for other form types supported
            
            if (ActionForm.class.isAssignableFrom(formFieldType)) {
                if (FORM_CONVERTER == null) {
                    FORM_CONVERTER = new FormConverter();
                }
                result = FORM_CONVERTER;
            }            
            
        }

        return result;
    }

    
    /**
     * <p>Return an object for use in converting the specified beanFieldType
     * to String and vice versa.</p>
     * 
     * @param beanFieldType the {@link Class} for the field on the bean
     * @return an object suitable for use as a converter between the  
     * beanFieldType and formFieldType.
     */
    protected Object getStringConverter(Class beanFieldType) {
        
        if (beanFieldType.equals(String.class)) {
            if (STRING_CONVERTER == null) {
                STRING_CONVERTER = new StringConverter();
            }
            return STRING_CONVERTER;
        }

        if ((beanFieldType.equals(Integer.class)) || (beanFieldType.equals(Integer.TYPE))
                || (beanFieldType.equals(Long.class)) || (beanFieldType.equals(Long.TYPE))
                || (beanFieldType.equals(Double.class)) || (beanFieldType.equals(Double.TYPE))
                || (beanFieldType.equals(Float.class)) || (beanFieldType.equals(Float.TYPE))
                || (beanFieldType.equals(Byte.class)) || (beanFieldType.equals(Byte.TYPE))
                || (beanFieldType.equals(Short.class)) || (beanFieldType.equals(Short.TYPE))) {
                if (NUMBER_CONVERTER == null) {
                    NUMBER_CONVERTER = new NumberConverter();
                }
                return NUMBER_CONVERTER;
        }

        if ((beanFieldType.equals(java.util.Date.class))
                || (beanFieldType.equals(java.sql.Date.class))
                || (beanFieldType.equals(java.sql.Timestamp.class))) {
                if (DATE_CONVERTER == null) {
                    DATE_CONVERTER = new DateConverter();
                }
                return DATE_CONVERTER;
        }
        
        if ((beanFieldType.equals(Boolean.class)) || (beanFieldType.equals(Boolean.TYPE))) {
            if (BOOLEAN_CONVERTER == null) {
                BOOLEAN_CONVERTER = new BooleanConverter();
            }
            return BOOLEAN_CONVERTER;
        }

        if ((beanFieldType.equals(Character.class)) || (beanFieldType.equals(Character.TYPE))) {
                if (CHAR_CONVERTER == null) {
                    CHAR_CONVERTER = new CharacterConverter();
                }
                return CHAR_CONVERTER;
        }

        return null;
    }

}
