package formdef.plugin.digester;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.digester.Digester;
import org.xml.sax.SAXException;

import java.io.InputStream;
import java.io.IOException;
import java.net.URL;

import formdef.plugin.config.FormDefConfig;

/**
 * @author Hubert Rabago
 */
public class DefaultDigester {

    protected static Log log = LogFactory.getLog(DefaultDigester.class);

    /**
     * The set of public identifiers, and corresponding resource names, for
     * the versions of the configuration file DTDs that we know about.
     */
    protected static String registrations[] = {
        "-//FormDef//FormDef Form Definition//EN",
        "/formdef/plugin/form-defs_0_5.dtd",
        "-//FormDef//FormDef Form Definition 0.6//EN",
        "/formdef/plugin/form-defs_0_6.dtd",
        "-//FormDef//FormDef Form Definition 1.0//EN",
        "/formdef/plugin/form-defs_1_0.dtd"
    };

    /**
     * Read the form configuration from the given input stream
     * @param input the input stream pointing to the form configuration
     * @return a {@link formdef.plugin.config.FormDefConfig} object containing the information read from the input stream
     * @throws java.io.IOException if an IO error occurs
     */
    public static FormDefConfig readConfiguration(InputStream input)
            throws IOException {
        FormDefConfig result = new FormDefConfig();
        readConfiguration(result, input);
        return result;
    }

    public static void readConfiguration(FormDefConfig result,
                                         InputStream input)
            throws IOException {
        log.debug("inside readConfiguration");

        Digester digester = new Digester();
        digester.push(result);
        digester.setNamespaceAware(true);  
        digester.setValidating(true);

        // register DTDs
        for (int i = 0; i < registrations.length; i += 2) {
            URL url = DefaultDigester.class.getResource(registrations[i+1]);
            if (url != null) {
                digester.register(registrations[i], url.toString());
            }
        }

        digester.setUseContextClassLoader(true);

        //*********************************************************************
        // Read formdef defaults
        digester.addSetProperties("form-definition/formdef-config");

        //*********************************************************************
        // Create global converter object
        digester.addObjectCreate("form-definition/global-converters/global-converter",
                formdef.plugin.config.GlobalConverterConfig.class);
        //digester.addCallMethod("form-definition/global-converters/global-converter","setForValue",1,new Class[]{String.class});
        //digester.addCallParam("form-definition/global-converters/global-converter",0,"for");
        digester.addSetProperties("form-definition/global-converters/global-converter",
                new String[] {"for","type","param","key","bundle"},
                new String[] {"forValueString","typeName","conversionParameter","conversionKey","conversionBundle"});
        digester.addSetNext("form-definition/global-converters/global-converter", "addConverter",
                "formdef.plugin.config.GlobalConverterConfig");

        //*********************************************************************
        // Create Form objects
        digester.addObjectCreate("form-definition/formset/form",
                "formdef.plugin.config.FormMappingConfig", "className");
        digester.addSetProperties("form-definition/formset/form",
                new String[] {"beanType","formType"},
                new String[] {"beanType","formType"});
        digester.addSetProperties("form-definition/formset/form/factory",
                new String[] {"type","method","nameParam"},
                new String[] {"factory","factoryMethodName","factoryMethodNameParam"});
        
        // use callMethod instead of setNestedProperties for "excludes"
        digester.addCallMethod("form-definition/formset/form/excludes",
                "setExcludes", 0, new Class[] {String.class});

        // Arbitrary properties for this form's FormBeanConfig
        digester.addCallMethod("form-definition/formset/form/set-property",
                "addConfigProperty", 2, new Class[] {String.class, String.class});
        digester.addCallParam("form-definition/formset/form/set-property",
                0, "key");
        digester.addCallParam("form-definition/formset/form/set-property",
                1, "value");
        
        digester.addSetNext("form-definition/formset/form", "addForm",
                "formdef.plugin.config.FormMappingConfig");

        //*********************************************************************
        // Create Field objects
        digester.addObjectCreate("form-definition/formset/form/field",
                "formdef.plugin.config.PropertyMappingConfig", "className");
        digester.addSetProperties("form-definition/formset/form/field",
                "property",
                "name");

        // Arbitrary properties for this field's FormPropertyConfig
        digester.addCallMethod("form-definition/formset/form/field/set-property",
                "addConfigProperty", 2, new Class[] {String.class, String.class});
        digester.addCallParam("form-definition/formset/form/field/set-property",
                0, "key");
        digester.addCallParam("form-definition/formset/form/field/set-property",
                1, "value");

        // Configure the field's converter
        digester.addSetProperties("form-definition/formset/form/field/converter",
                new String[] {"name","type","param","key","bundle"},
                new String[] {"converterName","converterType","conversionParam","conversionKey","conversionBundle"});

        // Arbitrary properties for this field's Converter
        digester.addCallMethod("form-definition/formset/form/field/converter/set-property",
                "addConverterProperty", 2, new Class[] {String.class, String.class});
        digester.addCallParam("form-definition/formset/form/field/converter/set-property",
                0, "key");
        digester.addCallParam("form-definition/formset/form/field/converter/set-property",
                1, "value");
        digester.addSetNext("form-definition/formset/form/field","addProperty",
                "formdef.plugin.config.PropertyMappingConfig");


        try {
            // Parse the input stream to initialize our database
            digester.parse(input);

        } catch (SAXException e) {
            log.error(e.getMessage(), e);
        }

    }

}
