package formdef.plugin.util;

import org.apache.struts.action.Action;
import org.apache.struts.action.ActionMapping;
import org.apache.struts.action.DynaActionForm;
import org.apache.struts.action.ActionForm;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.config.FormBeanConfig;
import org.apache.struts.util.RequestUtils;
import org.apache.struts.util.MessageResources;
import org.apache.struts.util.ModuleUtils;
import org.apache.struts.Globals;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;
import org.apache.commons.beanutils.DynaClass;
import org.apache.commons.beanutils.DynaProperty;
import org.apache.commons.beanutils.PropertyUtils;
import org.apache.commons.beanutils.DynaBean;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import javax.servlet.ServletContext;

import formdef.plugin.conversion.Converter;
import formdef.plugin.conversion.ConversionContext;
import formdef.plugin.conversion.FormConverter;
import formdef.plugin.conversion.FormCollectionConverter;
import formdef.plugin.config.FormDefConfig;
import formdef.plugin.FormMapping;
import formdef.plugin.PropertyMapping;

import java.util.Map;
import java.util.Locale;
import java.util.Collection;
import java.util.HashMap;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Constructor;

/**
 * <p>Utility methods for formatting form values from business objects and
 * vice versa.</p>
 *
 * @author Hubert Rabago
 * @author Joe Hertz
 * @author Umberto Nicoletti
 */
public class FormUtils {

    /** the logger for this class. */
    private static final Log log = LogFactory.getLog(FormUtils.class);

    /** the instance being returned by getInstance(). */
    protected static FormUtils instance = null;

    /** the classname of the factory to use in creating FormUtils instances. */
    protected static String factoryClassName =
            "formdef.plugin.util.FormUtilsFactoryImpl";

    /** 
     * the classname of the factory to use in creating 
     * ResourceLocaleUtil instances.
     */
    protected static String resourceLocaleUtilFactoryClassName =
            "formdef.plugin.util.ResourceLocaleUtilFactoryImpl";


    //**************************************************************************
    // public static methods

    
    /**
     * <p>Prepares the dyna form by acquiring an instance and populating it
     * with values from the provided source object.</p>
     *
     * @param formName the name of the formbean
     * @param source the object holding the values to be used for the formbean
     * @param action the current action
     * @param mapping the current ActionMapping
     * @param request the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      getter methods are inaccessible or another error was encountered
     *      during the formatting of form fields.
     */
    public static ActionForm setFormValues(
            String formName,
            Object source,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        return getInstance()
                .setFormValuesImpl(formName, source, action, mapping, request);
    }

    
    /**
     * <p>Create the object which contains the values held by the given
     * actionForm (actionForm must be associated with mapping)</p>
     *
     * @param actionForm the {@link ActionForm} to process
     * @param action the action currently executing
     * @param mapping the mapping to which the form was submitted
     * @param request the request being processed
     * 
     * @return an object holding the values from the actionForm
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public static Object getFormValues(
            ActionForm actionForm,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        return getInstance().getFormValuesImpl(actionForm, 
                action, mapping, request);
    }

    
    /**
     * <p>Create the object which contains the values held by the given
     * actionForm when actionForm is not one associated with mapping.</p>
     *
     * @param actionForm the {@link ActionForm} to process
     * @param formName the name of this actionForm
     * @param action the action currently executing
     * @param mapping the mapping to which the form was submitted
     * @param request the request being processed
     * 
     * @return an object holding the values from the actionForm
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public static Object getFormValues(
            ActionForm actionForm,
            String formName,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        return getInstance().getFormValuesImpl(actionForm, formName, 
                action, mapping, request);
    }
    

    /**
     * <p>Copy the values of the given <code>actionForm</code> to the given
     * <code>dest</code>.  <code>dest</code> must be an instance of the
     * class that is configured with the given <code>actionForm</code>.</p>
     *
     * @param actionForm the {@link ActionForm} to process
     * @param dest      the object which actionForm values will be copied to
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public static void getFormValues(
            ActionForm actionForm,
            Object dest,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        getInstance().getFormValuesImpl(actionForm, dest, 
                action, mapping, request);
    }

    
    /**
     * <p>Create the object which contains the values held by the given
     * actionForm when actionForm is not one associated with mapping.</p>
     *
     * @param actionForm the {@link ActionForm} to process
     * @param dest      the object which actionForm values will be copied to
     * @param formName  the name of this actionForm
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public static void getFormValues(
            ActionForm actionForm,
            Object dest,
            String formName,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        getInstance().getFormValuesImpl(actionForm, dest, formName, 
                action, mapping, request);
    }


    /**
     * <p>Find the name of the form mapping for the given object.  If the
     * given object has several mappings, only the first mapping found
     * is returned.</p>
     * 
     * <p>The result is cached in a map to avoid costly lookups.</p>
     *
     * @param object    the object whose form mapping is needed.
     * @return the name of the form mapping found for the given object.
     */
    public static String lookupFormName(Object object, 
                                        Action action, 
                                        ActionMapping mapping) {
        return getInstance().lookupFormNameImpl(object, action, mapping);
    }


    //**************************************************************************
    // methods used for overriding FormUtils.
    
    
    /**
     * <p>Get an instance of FormUtils.</p>
     *
     * @return a FormUtils instance that can be used for setting and getting
     * form values.
     */
    public static FormUtils getInstance() {
        if (instance == null) {
            FormUtilsFactory factory = getFormUtilsFactory();
            instance = factory.createFormUtils();
            if (log.isInfoEnabled()) {
                log.info("Setting FormUtils instance to " + instance);
            }
        }
        return instance;
    }


    /**
     * Set the name of the factory class that will be used to create
     * the actual FormUtils object.
     */
    public static void setFactoryClassName(String factoryClassName) {
        if (log.isDebugEnabled()) {
            log.debug("Setting factoryClassName to " + factoryClassName);
        }
        FormUtils.factoryClassName = factoryClassName;
    }


    /**
     * Set the name of the factory class that will be used to create
     * the actual FormUtils object.
     */
    public static void setResourceLocaleUtilFactoryClassName(
            String resourceLocaleUtilFactoryClassName) {
        if (log.isDebugEnabled()) {
            log.debug("Setting resourceLocaleUtilFactoryClassName to " 
                    + resourceLocaleUtilFactoryClassName);
        }
        FormUtils.resourceLocaleUtilFactoryClassName 
                = resourceLocaleUtilFactoryClassName;
    }


    protected static FormUtilsFactory getFormUtilsFactory() {
        FormUtilsFactory factory = null;
        try {
            factory = (FormUtilsFactory)
                    RequestUtils.applicationInstance(factoryClassName);
        } catch (Exception e) {
            log.error("Exception [" + e + "," + e.getMessage() + "]", e);

            String message =
                    "Unable to create FormUtilsFactory object: "
                    + factoryClassName;
            throw new IllegalArgumentException(message);
        }
        return factory;
    }


    /**
     * Get an instance of the ResourceLocaleUtilFactory configured for this
     * application.
     */ 
    protected ResourceLocaleUtilFactory getResourceLocaleUtilFactory() {
        ResourceLocaleUtilFactory factory = null;
        try {
            factory = (ResourceLocaleUtilFactory)
                    RequestUtils.applicationInstance(
                            resourceLocaleUtilFactoryClassName);
        } catch (Exception e) {
            log.error("Exception [" + e + "," + e.getMessage() + "]", e);

            String message =
                    "Unable to create ResourceLocaleUtilFactory object: "
                    + resourceLocaleUtilFactoryClassName;
            throw new IllegalArgumentException(message);
        }
        return factory;
    }

    
    //**************************************************************************
    // public utility methods

    
    /**
     * Return the FormDefConfig for a specific module.
     * <p/>
     * @param context   the current servlet context
     * @param moduleConfig  the moduleConfig for the current module
     * @return the FormDefConfig for the current module
     */ 
    public FormDefConfig getFormDefConfig(ServletContext context, 
                                          ModuleConfig moduleConfig) {
        FormDefConfig config =
                (FormDefConfig) context.getAttribute(
                        FormDefConfig.FORMDEF_KEY + moduleConfig.getPrefix());
        return config;
    }


    /**
     * Find the form definition identified by formName
     * @param formName the name of the form to be found
     * @param context the current servlet context
     * @param moduleConfig the current module
     * @return the {@link formdef.plugin.FormMapping} identified by formName
     */
    public FormMapping findFormDefinition(
            String formName,
            ServletContext context,
            ModuleConfig moduleConfig) {
        FormDefConfig config = getFormDefConfig(context, moduleConfig);
        FormMapping form = config.getForm(formName);
        return form;
    }

    
    /**
     * <p>Implementation of the {@link #setFormValues} static method.
     * Prepares the dyna form by acquiring an instance and populating it
     * with values from the provided source object.</p>
     *
     * @param formName the name of the formbean
     * @param source the object holding the values to be used for the formbean
     * @param action the current action
     * @param mapping the current ActionMapping
     * @param request the request being processed
     */
    public ActionForm setFormValuesImpl(
            String formName,
            Object source,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        //ServletContext context = action.getServlet().getServletContext();
        ModuleConfig moduleConfig = mapping.getModuleConfig();

        // create the ActionForm we'll use
        ActionForm form =
                createActionForm(formName, action, mapping, request);
        
        // for now, only DynaBean implementations are supported
        if (form instanceof DynaBean) {
            populateForm(formName, form, source,
                    moduleConfig, action, mapping, request);
        }

        return form;
    }



    /**
     * Sets the form's values from the source object using the
     * definitions associated with the given formName.
     * <p>
     * This method has been <b>deprecated</b> and does not support 
     * automatic handling of nested beans.
     * <p>
     * @param formName the name of the form definition linking the
     *    form with the source object
     * @param dynaForm the form to populate with values from the source object
     * @param source the object holding the values to be used for the form
     * @param action the current action
     * @param moduleConfig the configuration object for the current module
     * @param request the request being processed
     * 
     * @deprecated this method has been deprecated and will be removed 
     *          in a future release.
     */
    public void populateForm(
            String formName,
            DynaActionForm dynaForm,
            Object source,
            Action action,
            ModuleConfig moduleConfig,
            HttpServletRequest request) {
        
        populateForm(formName, dynaForm, source, moduleConfig, 
                action, null, request);
        
    }

    /**
     * Sets the form's values from the source object using the
     * definitions associated with the given formName.
     * <p>
     * @param formName the name of the form definition linking the
     *    form with the source object
     * @param actionForm the form to populate with values from the source object
     * @param source the object holding the values to be used for the form
     * @param action the current action
     * @param moduleConfig the configuration object for the current module
     * @param request the request being processed
     */
    public void populateForm(
            String formName,
            ActionForm actionForm,
            Object source,
            ModuleConfig moduleConfig,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        ServletContext context = action.getServlet().getServletContext();

        // find the form definition identified by formName
        FormMapping formMapping = 
                findFormDefinition(formName, context, moduleConfig);
        if (formMapping == null) {
            throw new IllegalArgumentException(
                    "Unable to find the definition for " + formName);
        }

        DynaClass dynaClass = null;
        if (actionForm instanceof DynaBean) {
            dynaClass = ((DynaBean) actionForm).getDynaClass();
        } else {
            throw new IllegalArgumentException(
                    formName + " is not a DynaBean implementation");
        }
        
        // copy property values from the source object to the form
        Map properties = formMapping.getProperties();
        java.util.Iterator iter = properties.keySet().iterator();
        while (iter.hasNext()) {
            String propertyName = (String) iter.next();
            
            // find our mapping info for this property
            PropertyMapping property =
                    (PropertyMapping) properties.get(propertyName);
            
            // get our converter
            Converter converter = (Converter) property.getConverter();
           
            if (converter == null) {
                if (log.isTraceEnabled()) {
                    log.trace("Skipping conversion of " + property.getName()
                            + " due to a null converter");
                }
                continue;
            }

            // make sure we can set this value
            if (property.getGetter() == null) {
                if (log.isTraceEnabled()) {
                    log.trace("Skipping conversion of " + property.getName()
                            + " due to a null getter");
                }
                continue;
            }

            // get the value for this property
            Object value = null;
            try {
                value = property.getGetter().invoke(source,new Object[]{});
            
                // get the type for this property on the form bean
                DynaProperty dynaProperty = 
                        dynaClass.getDynaProperty(propertyName);
                Class formType = dynaProperty.getType();
                
                // prepare our conversion context
                ConversionContext conversionContext = 
                        prepareConversionContext(source, actionForm, 
                                propertyName, value, formType, property, 
                                action, mapping, request);
    

                // format the value through the converter
                Object formattedValue = converter.convert (conversionContext);

                // pass it to the form
                PropertyUtils.setProperty(
                        actionForm, propertyName, formattedValue);

            } catch (RuntimeException e) {
                // rethrow RTEs for backward compatibility
                throw e;
            } catch (Exception e) {
                if (log.isErrorEnabled()) {
                    log.error(e + " was thrown during converion of " 
                        + propertyName, e);
                }
                throw new IllegalArgumentException();
            }
        }
    }

    
    /**
     * Create the ConversionContext that will be used for the given set of
     * parameters.
     * @param value     the value to be converted.
     * @param type      the type desired for the result.
     * @param request   the request being processed.
     * @param property  the property mapping for this value.
     * @param action    the action processing the request.
     * @return  the {@link ConversionContext} appropriate for this property
     *  and for this request.
     * @deprecated use the method that accepts an action mapping.
     */ 
    protected ConversionContext prepareConversionContext(
            Object value, 
            Class type, 
            HttpServletRequest request, 
            PropertyMapping property, 
            Action action) {
        ConversionContext conversionContext =
                new ConversionContext(value, type);
        conversionContext.setLocale(getLocale(request));
        populateConversionParameter(conversionContext, property, request, action);
        //conversionContext.setParam(conversionParam);
        return conversionContext;
    }


    /**
     * Create the ConversionContext that will be used for the given set of
     * parameters.
     * @param source    the source object whose fields are being converted.
     * @param destination   the object to holding all the converted fields.
     * @param propertyName  the name of the field to be converted.
     * @param value     the value of the field to be converted.
     * @param type      the type desired for the result of the field conversion.
     * @param property  the property mapping for this value.
     * @param action    the action processing the request.
     * @param action    the mapping the request was submitted to.
     * @param request   the request being processed.
     * @return  the {@link ConversionContext} appropriate for this property
     *  and for this request.
     */ 
    protected ConversionContext prepareConversionContext(
            Object source,
            Object destination,
            String propertyName,
            Object value, 
            Class type, 
            PropertyMapping property, 
            Action action,
            ActionMapping mapping, 
            HttpServletRequest request) {
        
        ConversionContext conversionContext =
                new ConversionContext(value, type);
        conversionContext.setSource(source);
        conversionContext.setDestination(destination);
        conversionContext.setPropertyName(propertyName);
        conversionContext.setLocale(getLocale(request));
        conversionContext.setAction(action);
        conversionContext.setMapping(mapping);
        conversionContext.setRequest(request);
        conversionContext.setProperties(property.getProperties());
        
        populateConversionParameter(conversionContext, property, request, action);
        //conversionContext.setParam(conversionParam);
        
        return conversionContext;
    }


    /**
     * Create the {@link DynaActionForm} instance 
     * identified by the given formName.
     * Uses the {@link FormBeanConfig#createActionForm}
     *  method introduced after Struts 1.2.0.
     * <p/>
     * <b>Note:</b>  While this method is not yet deprecated, it is recommended
     * that calls to this method be changed to use either 
     * {@link FormBeanConfig#createActionForm} or {@link #createActionForm}
     * instead.  
     * <p/>
     * @param formName the name used to identify the form
     * @param moduleConfig the configuration for the current module
     * @param action the current action mapping 
     * @return the DynaActionForm identified by formName
     */
    public DynaActionForm createDynaActionForm(
            String formName,
            ModuleConfig moduleConfig,
            Action action) {
        DynaActionForm dynaForm = null;
        try {
            FormBeanConfig formBeanConfig =
                    moduleConfig.findFormBeanConfig(formName);
            
            dynaForm = (DynaActionForm) 
                    formBeanConfig.createActionForm(action.getServlet());
        } catch (Exception e) {
            String message = "Unable to create the form bean for " + formName;
            log.error(message + " [" + e + "," + e.getMessage() + "]", e);
            
            // if you use JDK 1.4, you could pass along the cause to the RTE
            throw new RuntimeException(message);
        }
        return dynaForm;
    }


    /**
     * Create the {@link ActionForm} instance identified by the given 
     * formName.  If the form has a FormDef mapping configured, the form is
     * checked for nested forms.  Those forms, when found, are also created.
     * <p/>
     * Uses the {@link FormBeanConfig#createActionForm}
     *  method introduced after Struts 1.2.0.
     * <p/>
     * @param formName the name used to identify the form
     * @param action the current action mapping 
     * @param mapping the mapping to which the form was submitted
     * @param request the request being processed
     * @return the ActionForm identified by formName
     */
    public ActionForm createActionForm(
            String formName,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {
        ActionForm form = null;
        try {
            ModuleConfig moduleConfig = 
                    ModuleUtils.getInstance().getModuleConfig(request);
            FormBeanConfig formBeanConfig =
                    moduleConfig.findFormBeanConfig(formName);
            
            form = formBeanConfig.createActionForm(action.getServlet());
            
            // see if there's a FormDef config for this form
            FormMapping formMapping = findFormDefinition(formName,
                    action.getServlet().getServletContext(), moduleConfig);
            
            if (formMapping != null) {
                initializeNestedForms(form, formMapping, 
                        action, mapping, request);
            }
        } catch (Exception e) {
            String message = "Unable to create the form bean for " + formName;
            log.error(message + " [" + e + "," + e.getMessage() + "]", e);
            
            // if you use JDK 1.4, you could pass along the cause to the RTE
            throw new RuntimeException(message);
        }
        return form;
    }

    
    /**
     * <p>Loops through the FormDef PropertyMapping for the given form and 
     * checks (through the assigned converters) if there are nested forms.  
     * If any are found, those fields are initialized through 
     * <code>createActionForm</code>.  For a field containing a 
     * <i>collection</i> of forms, its {@link FormCollectionConverter} is 
     * used to initialize the collection.</p>
     *    
     * @param form          the form to inspect.
     * @param formMapping   the FormDef mapping for the given form.
     * @param action        the current action. 
     * @param mapping       the mapping to which the request was submitted.
     * @param request       the request being processed.
     */ 
    protected void initializeNestedForms(ActionForm form, 
                                         FormMapping formMapping, 
                                         Action action, 
                                         ActionMapping mapping, 
                                         HttpServletRequest request) 
            throws  IllegalAccessException, 
                    InvocationTargetException, 
                    NoSuchMethodException {
        
        // see if there are nested forms
        Map properties = formMapping.getProperties();
        java.util.Iterator iterator = properties.keySet().iterator();
        while (iterator.hasNext()) {
            PropertyMapping property = 
                    (PropertyMapping) properties.get(iterator.next());

            Object converter = property.getConverter();

            if (converter instanceof FormConverter) {
                // if the converter is a FormConverter, its param
                //      is the form name
                String name = property.getConversionParam();
                    
                if (name != null) {
                    // if we have the form's name, let's create it 
                    //  and initialize the field
                    ActionForm value = createActionForm(name,
                            action, mapping, request);
                    PropertyUtils.setProperty(form, 
                            property.getName(), value);
                }
            } else if (converter instanceof FormCollectionConverter) {
                // if we're using a collection converter for this field,
                //  use that converter to initialize the collection object
                //  for the property.
                FormCollectionConverter fcc = 
                        (FormCollectionConverter) converter;
                ConversionContext context = prepareConversionContext(null, 
                        form, property.getName(), null, 
                        property.getFormFieldType(), 
                        property, action, mapping, request);
                
                // use arbitrary properties to pass a flag to the converter
                //  to indicate that a collection is being created for
                //  initialization purposes
                context.setProperty(
                        FormCollectionConverter.PROP_INITIALIZE_FLAG_KEY,
                        new Boolean(true).toString());
                
                Collection value = fcc.getFormCollection(context, 
                        property.getName(), null);
                PropertyUtils.setProperty(form, property.getName(), value);
            }
                
        }
    }


    /**
     * <p>Implementation of the {@link #getFormValues} static method.
     * Creates the object which contains the values held by the given
     * actionForm.</p>
     *
     * @param actionForm the {@link ActionForm} that holds the submitted values
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @return an object holding the values from the actionForm
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public Object getFormValuesImpl(
            ActionForm actionForm,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        // find the form definition we'll be using
        String formName = mapping.getName();

        return this.getFormValuesImpl(actionForm, formName, 
                action, mapping, request);
    }

    
    /**
     * <p>Implementation of the {@link #getFormValues} static method.
     * Creates the object which contains the values held by the given
     * actionForm.</p>
     *
     * @param actionForm the {@link ActionForm} that holds the submitted values
     * @param dest      the object which actionForm values will be copied to
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public void getFormValuesImpl(
            ActionForm actionForm,
            Object dest,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        // find the form definition we'll be using
        String formName = mapping.getName();

        this.getFormValuesImpl(actionForm, dest, formName, 
                action, mapping, request);
    }

    
    /**
     * <p>Implementation of the {@link #getFormValues} static method.
     * Returns the object which will contain the values held by the given
     * actionForm.</p>
     *
     * @param actionForm the {@link ActionForm} that holds the submitted values
     * @param formName  identifies the particular actionForm.
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public Object getFormValuesImpl(
            ActionForm actionForm,
            String formName,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        ServletContext context = action.getServlet().getServletContext();
        ModuleConfig moduleConfig = mapping.getModuleConfig();

        // find the form definition we'll be using
        FormMapping formMapping = 
                findFormDefinition(formName, context, moduleConfig);
        if (formMapping == null) {
            throw new IllegalArgumentException(
                    "Unable to find the definition for " + formName);
        }

        // create the object for this form
        Object populatedObject = createObjectInstance(formMapping);
        
        getFormValuesImpl(actionForm, populatedObject, formName, 
                action, mapping, request);
        
        return populatedObject;
    }
    
    
    /**
     * <p>Implementation of the {@link #getFormValues} static method.
     * Retrieves the values held by the given <code>actionForm</code> and
     * copies them to the given <code>dest</code> object.</p>
     *
     * @param actionForm the {@link ActionForm} that holds the submitted values
     * @param dest      the object which actionForm values will be copied to
     * @param formName  identifies the particular actionForm.
     * @param action    the action currently executing
     * @param mapping   the mapping to which the form was submitted
     * @param request   the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's constructor, factory, or setter methods are inaccessible
     *      or throws an exception when called.
     */
    public void getFormValuesImpl(
            ActionForm actionForm,
            Object dest,
            String formName,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        if (!(actionForm instanceof DynaBean)) {
            throw new IllegalArgumentException(
                    formName + " is not a DynaBean implementation.");
        }

        ServletContext context = action.getServlet().getServletContext();
        ModuleConfig moduleConfig = mapping.getModuleConfig();

        // find the form definition we'll be using
        FormMapping formMapping = 
                findFormDefinition(formName, context, moduleConfig);
        if (formMapping == null) {
            throw new IllegalArgumentException(
                    "Unable to find the definition for " + formName);
        }
        
        // verify the type of 'dest'
        Class beanType = formMapping.getBeanType();
        if (!beanType.isInstance(dest)) {
            throw new IllegalArgumentException(
                    "The given destination object is not an instance of "
                    + beanType.getName());
        }

        // populate our object with values from the actionForm
        populateBeanFromForm(formMapping, dest, actionForm, 
                action, mapping, request);
    }

    
    /**
     * Creates an instance of the object described by the given form.
     * 
     * @param form a {@link formdef.plugin.FormMapping} instance describing the 
     *              object to create
     * @return a new instance of the object described by form as returned
     *   by its no-arg constructor or its specified factory
     * @throws java.lang.IllegalArgumentException if access modifiers for the 
     *      object's no-arg constructor or its designated factory method make 
     *      it inaccessible to this routine or the factory throws an exception.
     */
    public Object createObjectInstance(FormMapping form) {

        if (form == null) {
            throw new IllegalArgumentException("form param cannot be null.");
        }

        Object populatedObject = null;

        // first, see if we should be using a factory
        if (form.getFactoryMethod() != null) {
            // it seems we are; should be pass the form name to it?
            Object[] params = null;
            if (form.isFactoryMethodNameParam()) {
                // yes, let the factory know what we're instantiating for
                params = new Object[] {form.getName()};
            } else {
                params = new Object[] {};
            }

            // invoke the factory method
            try {
                populatedObject = form.
                        getFactoryMethod().invoke(form.getFactory(),params);
                
            } catch (RuntimeException e) {
                // rethrow RTEs for backward compatibility
                throw e;
            } catch (Exception e) {
                log.error(e + " was thrown while calling bean factory "
                    + " for " + form.getName(), e);
                throw new IllegalArgumentException();
            }

        } else {

            try {

                // find and call the no-arg constructor
                Constructor constructor =
                        form.getBeanType().getConstructor(new Class[]{});

                populatedObject = constructor.newInstance(new Object[]{});

            } catch (NoSuchMethodException e) {
                log.error("Cannot find no-arg constructor", e);
                throw new IllegalArgumentException("Cannot find"
                    + " no-arg constructor for " + form.getName());
            } catch (RuntimeException e) {
                throw e;
            } catch (Exception e) {
                log.error("Exception creating object instance for "
                        + form.getName(), e);
                throw new IllegalArgumentException(
                        "Unable to instantiate " + form.getName());
            }
        }
        return populatedObject;
    }


    /**
     * Populates the dest JavaBean with values coming from an ActionForm src.
     * @param form the form definition object
     * @param dest the destination object which will hold the values
     * @param src a populated {@link ActionForm}
     * @param action the action currently executing
     * @param request the request being processed
     * @throws java.lang.IllegalAccessException if the dest setter methods are
     *      inaccessible
     * @throws java.lang.reflect.InvocationTargetException if the dest setter 
     *      methods throw an exception
     * 
     * @deprecated this method will be removed in a future release.
     *      Use the populateBeanFromForm method that accepts
     *      an ActionMapping parameter. 
     */
    public void populateBeanFromForm(
            FormMapping form,
            Object dest,
            ActionForm src,
            HttpServletRequest request,
            Action action)
                throws IllegalAccessException, InvocationTargetException {
        
        populateBeanFromForm(form, dest, src, action, null, request); 
        
    }

    /**
     * Populates the dest JavaBean with values coming from an ActionForm src.
     * 
     * @param form the form definition object
     * @param dest the destination object which will hold the values
     * @param src a populated {@link ActionForm}
     * @param action the action currently executing
     * @param request the request being processed
     * 
     * @throws java.lang.IllegalArgumentException if the dest setter methods are
     *      inaccessible or throws an exception
     */
    public void populateBeanFromForm(
            FormMapping form,
            Object dest,
            ActionForm src,
            Action action,
            ActionMapping mapping,
            HttpServletRequest request) {

        // get the list of properties for this form
        Map properties = form.getProperties();

        // loop through all properties
        java.util.Iterator iter = properties.keySet().iterator();
        while (iter.hasNext()) {
            String propertyName = (String) iter.next();

            PropertyMapping property =
                    (PropertyMapping) properties.get(propertyName);
            Converter converter = (Converter) property.getConverter();
           
            if (converter == null) {
                if (log.isTraceEnabled()) {
                    log.trace("Skipping conversion of " + property.getName()
                            + " due to a null converter");
                }
                continue;
            }

            if (property.getSetter() == null) {
                if (log.isTraceEnabled()) {
                    log.trace("Skipping conversion of " + property.getName()
                            + " due to a null setter");
                }
                continue;
            }

            // get the formatted value from the form
            Object formattedValue = null;
            try {
                formattedValue = PropertyUtils.getProperty(src, propertyName);

                // parse it using the converter
                ConversionContext conversionContext = 
                        prepareConversionContext(src, dest, propertyName,
                                formattedValue, property.getType(), property, 
                                action, mapping, request);

                
                Object value = converter.convert (conversionContext);

                // pass the value to the dest object
                property.getSetter().invoke(dest, new Object[]{value});

            } catch (RuntimeException e) {
                // rethrow RTEs for backward compatibility
                throw e;
            } catch (Exception e) {
                log.error("Unable to populate the bean for " + form.getName(), 
                        e);
                throw new IllegalArgumentException(
                        "Unable to populate the bean for " + form.getName());
            }
        }
    }


    /**
     * Find the Class object with the given class name.
     */
    public Class classForName(String className)
            throws ClassNotFoundException {
        Class result = null;
        ClassNotFoundException caught = null;
        try {
            result = Class.forName(className);
        } catch (ClassNotFoundException e) {
            caught = e;
        }
        if (result == null) {
            // check if we're looking for a primitive type
            if (className.equals(Boolean.TYPE.getName())) {
                return Boolean.TYPE;
            } else if (className.equals(Byte.TYPE.getName())) {
                return Byte.TYPE;
            } else if (className.equals(Character.TYPE.getName())) {
                return Character.TYPE;
            } else if (className.equals(Double.TYPE.getName())) {
                return Double.TYPE;
            } else if (className.equals(Float.TYPE.getName())) {
                return Float.TYPE;
            } else if (className.equals(Integer.TYPE.getName())) {
                return Integer.TYPE;
            } else if (className.equals(Long.TYPE.getName())) {
                return Long.TYPE;
            } else if (className.equals(Short.TYPE.getName())) {
                return Short.TYPE;
            } else {
                if (caught != null) {
                    throw caught;
                } else {
                    throw new ClassNotFoundException(
                            "Unable to find class " + className);
                }
            }
        }
        return result;
    }


    /**
     * Populate the conversion parameter and locale to be used for the given 
     * property.
     * @param property the property whose conversion parameter is needed
     * @param request the request being processed
     * @param action the action being executed
     */ 
    protected void populateConversionParameter(
            ConversionContext conversionContext,
            PropertyMapping property,
            HttpServletRequest request,
            Action action) {
        
        // is the conversion parameter directly specified?
        String conversionParameter = property.getConversionParam();
        if ((conversionParameter == null) 
                || (conversionParameter.length() == 0)) {
            // is the key and/or bundle specified
            String key = property.getConversionKey();
            String bundle = property.getConversionBundle();
            
            // check if the key is populated
            if ((key != null) && (key.length() > 0)) {
                // make sure the bundle has a valid value
                if ((bundle != null) && (bundle.length() == 0)) {
                    bundle = null;
                }
                
                ServletContext servletContext = 
                        action.getServlet().getServletContext();
                
                // find the conversion parameter from the message resources
                conversionParameter = evaluateMessage( key, bundle, request, 
                        servletContext );
                
                // find the locale to use for this context
                ResourceLocaleUtil resourceLocaleUtil =
                        getResourceLocaleUtilFactory().getLocaleUtil(
                                request, servletContext, bundle);                
                Locale actualLocale = 
                        resourceLocaleUtil.getLocale(
                                conversionContext.getLocale(), key);
                
                conversionContext.setLocale(actualLocale);
            }
        }
        
        conversionContext.setParam(conversionParameter);
    }


    /**
     * Get the locale that will be used for a conversion operation.
     */
    protected Locale getLocale(HttpServletRequest request) {
        Locale locale = null;

        // see if there's a locale present among the session attributes
        HttpSession session = request.getSession(false);
        if (session != null) {
            locale = (Locale) session.getAttribute(Globals.LOCALE_KEY);
        }

        // if we can't find anything in the session,
        //      use the one which request provides
        if (locale == null) {
            locale = request.getLocale();
        }

        return locale;
    }

    /**
     * <p>Evaluate the given message resource key.</p>
     *
     * @param messageKey the name of the resource entry to retrieve
     * @param bundle The key specified in the
     *  <code>&lt;message-resources&gt;</code> element for the
     *  resource bundle to use
     * @param request the request being processed
     * @param servletContext the current {@link ServletContext}
     * @return the value of paramName from the message resources
     */
    protected String evaluateMessage(
            String messageKey,
            String bundle,
            HttpServletRequest request,
            ServletContext servletContext) {
        // get the message resources
        MessageResources resources;
        if (bundle != null) {
            resources = getResources(bundle, request, servletContext);
        } else {
            resources = getResources(request);
        }
        // evaluate the message
        String messageString = getMessage(messageKey, null, request,
                resources);

        // add it to the redirect parameters
        if ((messageString != null) && (messageString.length() > 0)) {
            return messageString;
        }
        return null;
    }


    /**
     * <p>Look for the ActionMessage in the provided MessageResources.</p>
     *
     * @param messageKey the key to look for
     * @param args the arguments to be used in evaluating the message
     * @param request the request being processed
     * @param resources the application's message resources
     * @return the message formed from the given ActionMessage
     */
    protected String getMessage(String messageKey,
                                Object[] args,
                                HttpServletRequest request,
                                MessageResources resources) {
        if (messageKey == null) {
            //log.warn("Call to getMessage() with a null messageKey.");
            return null;
        }

        String result = null;
        try {
            // first, get the locale from the request
            Locale userLocale = getLocale(request);

            // then, get the message
            if (args == null) {
                result = (resources.getMessage(userLocale, messageKey));
            } else {
                result = (resources.getMessage(userLocale, messageKey,
                        args));
            }
        } catch (Exception e) {
            //log.error("Exception while looking for message.", e);
        }
        return result;
    }

    /**
     * <p>Return the default message resources for the current module.</p>
     *
     * @param request The servlet request we are processing
     */
    protected MessageResources getResources(HttpServletRequest request) {

        return ((MessageResources)
                request.getAttribute(Globals.MESSAGES_KEY));

    }

    /**
     * <p>Return the specified message resources for the current module.</p>
     *
     * @param bundle The key specified in the
     *  <code>&lt;message-resources&gt;</code> element for the
     *  requested bundle
     * @param request The servlet request we are processing
     * @param servletContext the current {@link ServletContext}
     */
    protected MessageResources getResources(String bundle,
                                            HttpServletRequest request,
                                            ServletContext servletContext) {

        // Identify the current module
        ModuleConfig moduleConfig =
                ModuleUtils.getInstance().getModuleConfig(
                        request, servletContext);

        // Return the requested message resources instance
        return ((MessageResources) servletContext.getAttribute
                (bundle + moduleConfig.getPrefix()));

    }
    
    
    /** 
     * The cache of results of calls to {@link #lookupFormNameImpl}. 
     */ 
    protected Map formNames = new HashMap();
    
    
    /**
     * <p>Find the name of the form mapping for the given object.  If the
     * given object has several mappings, only the first mapping found
     * is returned.</p>
     * 
     * <p>The result is cached in a map to avoid costly lookups.</p>
     *
     * @param object    the object whose form mapping is needed.
     * @return the name of the form mapping found for the given object.
     */
    public String lookupFormNameImpl(Object object,
                                     Action action,
                                     ActionMapping mapping) {
        String formName = null;

        ModuleConfig moduleConfig = mapping.getModuleConfig();
        String objectName = object.getClass().getName();

        // form the key we used/would use to cache this
        String key = moduleConfig.getPrefix() + objectName;
        
        // if we have this cached, let's use that value
        if (formNames.containsKey(key)) {
            return (String) formNames.get(key);
        }

        // no cache yet; let's look for it in this module's formdef config
        ServletContext servletContext = action.getServlet().getServletContext();
        FormDefConfig config = getFormDefConfig(servletContext, moduleConfig);

        for (int i = 0; i < config.getFormNames().length; i++) {
            FormMapping formMapping = config.getForm(config.getFormNames()[i]);
            if (formMapping.getBeanType().getName().equals(objectName)) {
                if (log.isDebugEnabled()) {
                    log.debug("Found mapping for: " + objectName
                            + " it is: " + formMapping.getName());
                }
                formName = formMapping.getName();
                break;
            }
        }
        
        // if we found it, let's cache it
        if (formName != null) {
            formNames.put(key, formName);
        }

        return formName;
    }

}
