package formdef.plugin.util;

import org.apache.struts.util.ModuleUtils;
import org.apache.struts.util.MessageResources;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.Globals;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletContext;
import java.util.HashMap;
import java.util.Map;
import java.util.Collections;

/**
 * Default implementation for the {@link ResourceLocaleUtilFactory}.
 * Caches ResourceLocaleUtil instances by module prefix and bundle,
 * and returns PropertyResourceLocaleUtil objects.
 */
public class ResourceLocaleUtilFactoryImpl 
        implements ResourceLocaleUtilFactory {
    
    /**
     * Cache of ResourceLocaleUtil objects for a given module/bundle combo.
     */ 
    protected static Map rlufMap = Collections.synchronizedMap(new HashMap()); 

    /**
     * Provide the ResourceLocaleUtil that will be used for conversion
     * parameters in the named resource bundle for the given module.
     * <p/>
     * @param request the request being processed
     * @param servletContext the current {@link javax.servlet.ServletContext}
     * @param bundle the bundle key, can be null for the default bundle 
     * @return the {@link ResourceLocaleUtil} instance to use for the given
     *  module and bundle
     */ 
    public ResourceLocaleUtil getLocaleUtil(
            HttpServletRequest request, 
            ServletContext servletContext,
            String bundle) {
        
        String key = getCacheKey(request, servletContext, bundle);

        // see if the rlu already exists
        Object result = rlufMap.get(key);
        if (result == null) {
            result = createResourceLocaleUtil(key, 
                    request, servletContext, bundle);

            // keep it in the cache
            rlufMap.put(key, result);
        }
        
        return (ResourceLocaleUtil) result;
    }


    /**
     * Create the ResourceLocaleUtil instance that will be used whenever 
     * the given key is used.
     * <p/> 
     * @param key the key that will be used for caching the object this method
     *          will return 
     * @param request the request being processed
     * @param servletContext the current {@link javax.servlet.ServletContext}
     * @param bundle the bundle key, can be null for the default bundle 
     * @return the ResourceLocaleUtil to use with the given parameters
     */ 
    protected ResourceLocaleUtil createResourceLocaleUtil(
            String key,
            HttpServletRequest request, 
            ServletContext servletContext, 
            String bundle) {
        
        if (bundle == null) {
            bundle = "";
        }

        ResourceLocaleUtil result;
        
        // get the config for this bundle
        MessageResources resources = null; 
        if ("".equals(bundle)) { 
            resources = getResources(request);
        } else {
            resources = getResources(bundle, request, servletContext);
        }
        
        ModuleConfig moduleConfig = 
                ModuleUtils.getInstance().getModuleConfig(
                        request, servletContext);
            
        // create the resource locale util
        result = new PropertyResourceLocaleUtil(servletContext, 
                moduleConfig, resources.getConfig());
        return result;
    }


    /**
     * Generate the key that will be used to cache the ResourceLocaleUtil
     * for the given parameters.
     * <p/> 
     * This implementation uses a key composed of the current module prefix
     * and the bundle key.
     * <p/>
     * @param request the request being processed
     * @param servletContext the current {@link javax.servlet.ServletContext}
     * @param bundle the bundle key, can be null for the default bundle 
     * @return the map key to use for this request parameter combination
     */ 
    protected String getCacheKey(
            HttpServletRequest request,
            ServletContext servletContext,
            String bundle) {
        
        if (bundle == null) {
            bundle = "";
        }
        
        ModuleConfig moduleConfig = 
                ModuleUtils.getInstance().getModuleConfig(request);
        
        // get the module name
        String moduleName = moduleConfig.getPrefix();
        
        // get the hash map key for the requested bundle's RLU
        String key = moduleName + "_" + bundle;
        return key;
    }


    /**
     * <p>Return the default message resources for the current module.</p>
     *
     * @param request The servlet request we are processing
     */
    protected MessageResources getResources(HttpServletRequest request) {

        return ((MessageResources)
                request.getAttribute(Globals.MESSAGES_KEY));

    }

    
    /**
     * <p>Return the specified message resources for the current module.</p>
     *
     * @param bundle The key specified in the
     *  <code>&lt;message-resources&gt;</code> element for the
     *  requested bundle
     * @param request The servlet request we are processing
     * @param servletContext the current {@link javax.servlet.ServletContext}
     */
    protected MessageResources getResources(String bundle,
                                            HttpServletRequest request,
                                            ServletContext servletContext) {

        // Identify the current module
        ModuleConfig moduleConfig =
                ModuleUtils.getInstance().getModuleConfig(
                        request, servletContext);

        // Return the requested message resources instance
        return ((MessageResources) servletContext.getAttribute
                (bundle + moduleConfig.getPrefix()));

    }
    
}
