package formdef.plugin.config;

import junit.framework.TestCase;
import junit.framework.TestSuite;
import org.apache.struts.config.ModuleConfig;
import org.apache.struts.config.ModuleConfigFactory;
import org.apache.struts.config.FormBeanConfig;
import org.apache.struts.config.FormPropertyConfig;
import org.apache.struts.action.DynaActionForm;
import org.apache.struts.validator.DynaValidatorForm;
import formdef.plugin.sample.SampleBean1;
import formdef.plugin.sample.SubSqlDate;
import formdef.plugin.sample.SampleBean2;
import formdef.plugin.sample.SubActionForm;
import formdef.plugin.sample.SampleIndexedBeanA;
import formdef.plugin.FormMapping;

import java.util.List;
import java.util.Iterator;

/**
 * @author Hubert Rabago
 */
public class TestFormDefConfig extends TestCase {

    //*****************************
    // the usual method suspects

    public TestFormDefConfig(String s) {
        super(s);
    }

    public static TestSuite getSuite() {
        return new TestSuite(TestFormDefConfig.class);
    }

    public static void main(String[] args) {
        junit.textui.TestRunner runner = new junit.textui.TestRunner();
        runner.doRun(TestFormDefConfig.getSuite());
    }



    //*****************************

    protected GlobalConverterConfig mdyConverter = null;
    protected GlobalConverterConfig propertyNamedConverter = null;
    protected GlobalConverterConfig namedConverter = null;
    protected FormMappingConfig sampleBean1FormConfig = null;
    protected FormMappingConfig sampleBean2FormConfig = null;
    protected FormMappingConfig sampleIndexedBeanAFormConfig = null;
    protected FormMappingConfig dynaForm1Config = null;
    protected FormMappingConfig subActionForm1Config = null;

    /** the instance used for tests that do not change the configuration */
    protected FormDefConfig testConfig = null;

    public void setUp()
            throws Exception {
        super.setUp();

        // Configure a global M/d/yyyy date converter
        mdyConverter = new GlobalConverterConfig();
        mdyConverter.setForValue(GlobalConverterConfig.FOR_PROPERTY_TYPE);
        mdyConverter.setTarget(java.util.Date.class.getName());
        mdyConverter.setConversionParameter("M/d/yyyy");
        mdyConverter.setExactMatch(false);

        // Configure a global converter for a property name
        propertyNamedConverter = new GlobalConverterConfig();
        propertyNamedConverter.setForValue(
                GlobalConverterConfig.FOR_PROPERTY_NAME);
        propertyNamedConverter.setTarget("fDouble");
        propertyNamedConverter.setConversionParameter("#####0.00");

        // Configure a global named converter
        namedConverter = new GlobalConverterConfig();
        namedConverter.setForValue(
                GlobalConverterConfig.FOR_CONVERTER_NAME);
        namedConverter.setTarget("namedConverter");
        namedConverter.setConversionParameter("###,##0.00");

        // Configure a couple of FormDefFormConfigs
        sampleBean1FormConfig =
                new FormMappingConfig("sampleBean1", SampleBean1.class.getName());

        sampleBean2FormConfig = 
                new FormMappingConfig("sampleBean2", SampleBean2.class.getName());
        
        sampleIndexedBeanAFormConfig = 
                new FormMappingConfig("sampleIndexedBeanA", 
                        SampleIndexedBeanA.class.getName());
        
        // Configure a form not tied to a bean
        dynaForm1Config = new FormMappingConfig();
        dynaForm1Config.setName("dynaForm1Config");
        dynaForm1Config.setFormType(DynaActionForm.class.getName());
        PropertyMappingConfig prop = new PropertyMappingConfig();
        prop.setName("firstName");
        dynaForm1Config.addProperty(prop);
        prop = new PropertyMappingConfig();
        prop.setName("lastName");
        dynaForm1Config.addProperty(prop);
        
        // create a prop with an initial value
        prop = new PropertyMappingConfig();
        prop.setName("sex");
        prop.setInitial("M");
        prop.setReset("true");
        dynaForm1Config.addProperty(prop);
        
        // create a prop with a size and type value
        prop = new PropertyMappingConfig();
        prop.setName("languages");
        prop.setType("java.lang.String[]");
        prop.setSize("5");
        dynaForm1Config.addProperty(prop);
        
        // Configure an ActionForm subclass that's not a dynaForm
        subActionForm1Config = new FormMappingConfig();
        subActionForm1Config.setName("subActionForm1Config");
        subActionForm1Config.setFormType(SubActionForm.class.getName());

        // Configure a FormDefConfig that can be shared among tests that
        //  do not change the config
        testConfig = new FormDefConfig();
        testConfig.addConverter(mdyConverter);
        testConfig.addConverter(propertyNamedConverter);
        testConfig.addConverter(namedConverter);
        testConfig.addForm(sampleBean1FormConfig);
        testConfig.addForm(sampleBean2FormConfig);
        testConfig.addForm(sampleIndexedBeanAFormConfig);
    }

    /**
     * Test adding a property type converter
     */
    public void testAddGlobalPropertyTypeConverter()
            throws Exception {
        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add a property type converter
        formDefConfig.addConverter(mdyConverter);

        // make sure it got added to the right map
        assertSame("Can't find the newly added converter",
                mdyConverter,
                formDefConfig.propertyTypeConverters.get(
                        mdyConverter.getTarget()));
    }

    /**
     * Test adding a property name converter
     */
    public void testAddGlobalPropertyNameConverter()
            throws Exception {
        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add a property name converter
        formDefConfig.addConverter(propertyNamedConverter);

        // make sure it got added to the right map
        assertSame("Can't find the newly added converter",
                propertyNamedConverter,
                formDefConfig.propertyNameConverters.get(
                        propertyNamedConverter.getTarget()));
    }

    /**
     * Test adding a named converter
     */
    public void testAddGlobalConverterNameConverter()
            throws Exception {
        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add a named converter
        formDefConfig.addConverter(namedConverter);

        // make sure it got added to the right map
        assertSame("Can't find the newly added converter",
                namedConverter,
                formDefConfig.converterNameConverters.get(
                        namedConverter.getTarget()));
    }

    /**
     * Test adding a form
     */
    public void testAddForm()
            throws Exception {

        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add the form
        formDefConfig.addForm(sampleBean1FormConfig);

        // check that the form config was added
        assertSame("FormConfig was not added",
                sampleBean1FormConfig,
                formDefConfig.formDefFormConfigs.get("sampleBean1"));

        // check that the FormDef form was added
        assertNotNull("Corresponding FormMapping was not added",
                formDefConfig.formDefForms.get("sampleBean1"));
    }

    /**
     * Test the findPropertyTypeConverter to see if it can find
     * exact type matches
     */
    public void testFindPropertyTypeConverterExactMatch()
            throws Exception {

        GlobalConverterConfig actual =
                testConfig.findPropertyTypeConverter(java.util.Date.class);

        assertSame("Incorrect converter found", mdyConverter, actual);
    }

    /**
     * Test the findPropertyTypeConverter to see if it can find
     * matches for subclasses
     */
    public void testFindPropertyTypeConverterSubclassMatch()
            throws Exception {

        // try one sublcass level
        GlobalConverterConfig actual =
                testConfig.findPropertyTypeConverter(java.sql.Date.class);
        assertSame("Incorrect converter found", mdyConverter, actual);

        // try two subclass levels
        actual = testConfig.findPropertyTypeConverter(SubSqlDate.class);
        assertSame("Incorrect converter found", mdyConverter, actual);
    }

    /**
     * Test the findPropertyTypeConverter to see if it can find
     * matches for superclasses
     */
    public void testFindPropertyTypeConverterSuperclassNonMatch()
            throws Exception {

        // try one sublcass level
        GlobalConverterConfig actual =
                testConfig.findPropertyTypeConverter(java.lang.Object.class);
        assertNull("Invalid converter found", actual);
    }

    /**
     * Test the findPropertyTypeConverter to see if it can find
     * matches for subclasses in the order they were configured
     */
    public void testFindPropertyTypeConverterOrderedMatch()
            throws Exception {

        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add two converters that can match java.sql.Timestamp
        //  start with the java.util.Date mdyConverter
        formDefConfig.addConverter(mdyConverter);

        //  then add a java.sql.Date converter
        GlobalConverterConfig sqlDateConverter = new GlobalConverterConfig();
        sqlDateConverter.setForValue(GlobalConverterConfig.FOR_PROPERTY_TYPE);
        sqlDateConverter.setTarget(java.sql.Date.class.getName());
        sqlDateConverter.setConversionParameter("yyyyMMdd");
        sqlDateConverter.setExactMatch(false);
        formDefConfig.addConverter(sqlDateConverter);

        // we should get the mdyConverter, which was added first
        GlobalConverterConfig actual =
                formDefConfig.findPropertyTypeConverter(SubSqlDate.class);
        assertSame("Incorrect converter found", mdyConverter, actual);

        // now create a config with the converters in another order
        formDefConfig = new FormDefConfig();
        formDefConfig.addConverter(sqlDateConverter);
        formDefConfig.addConverter(mdyConverter);

        // we should get the sqlDateConverter
        actual = formDefConfig.findPropertyTypeConverter(SubSqlDate.class);
        assertSame("Incorrect converter found", sqlDateConverter, actual);
    }

    /**
     * Test the findPropertyTypeConverter to see if it can find
     * matches for subclasses in the order they were configured
     * while skipping converters that require an exact match
     */
    public void testFindPropertyTypeConverterExactOrderedMatch()
            throws Exception {

        // create the instance we're testing
        FormDefConfig formDefConfig = new FormDefConfig();

        // add two converters that can match SubSqlDate

        //  start with the java.util.Date converter
        GlobalConverterConfig utilDateConverter = new GlobalConverterConfig();
        utilDateConverter.setForValue(GlobalConverterConfig.FOR_PROPERTY_TYPE);
        utilDateConverter.setTarget(java.util.Date.class.getName());
        utilDateConverter.setConversionParameter("MM/dd/yyyy");
        formDefConfig.addConverter(utilDateConverter);

        // configure the utilDateConverter to require an exact match
        utilDateConverter.setExactMatch(true);

        //  then add a java.sql.Date converter
        GlobalConverterConfig sqlDateConverter = new GlobalConverterConfig();
        sqlDateConverter.setForValue(GlobalConverterConfig.FOR_PROPERTY_TYPE);
        sqlDateConverter.setTarget(java.sql.Date.class.getName());
        sqlDateConverter.setConversionParameter("yyyyMMdd");
        sqlDateConverter.setExactMatch(false);
        formDefConfig.addConverter(sqlDateConverter);


        // we should get the sqlDateConverter
        GlobalConverterConfig actual =
                formDefConfig.findPropertyTypeConverter(SubSqlDate.class);
        assertSame("Incorrect converter found", sqlDateConverter, actual);

        // now create a config with the converters in another order
        formDefConfig = new FormDefConfig();
        formDefConfig.addConverter(sqlDateConverter);
        formDefConfig.addConverter(utilDateConverter);

        // clear the exactMatch flag of utilDateConverter, then set
        //  that of the sqlDateConverter
        utilDateConverter.setExactMatch(false);
        sqlDateConverter.setExactMatch(true);

        // we should get the utilDateConverter
        actual = formDefConfig.findPropertyTypeConverter(SubSqlDate.class);
        assertSame("Incorrect converter found", utilDateConverter, actual);
    }

    /**
     * Test findPropertyNameConverter
     */
    public void testFindPropertyNameConverterExactMatch()
            throws Exception {

        GlobalConverterConfig actual =
                testConfig.findPropertyNameConverter(
                        propertyNamedConverter.getTarget());

        assertSame("Incorrect converter found",
                propertyNamedConverter, actual);
    }

    /**
     * Test findConverterNameConverter
     */
    public void testFindConverterNameConverterExactMatch()
            throws Exception {

        GlobalConverterConfig actual =
                testConfig.findConverterNameConverter(
                        namedConverter.getTarget());

        assertSame("Incorrect converter found", namedConverter, actual);
    }

    /**
     * Test getForm
     */
    public void testGetForm()
            throws Exception {

        FormMapping form = testConfig.getForm(sampleBean1FormConfig.getName());
        assertEquals("Incorrect form returned",
                form.getName(), sampleBean1FormConfig.getName());
    }
    
    /**
     * Check that the forms are being added to the module config 
     */ 
    public void testRegisterForms() 
            throws Exception {
        
        // create the instance we're testing and add a form to it
        FormDefConfig formDefConfig = new FormDefConfig();
        formDefConfig.addForm(sampleBean1FormConfig);
        formDefConfig.addForm(sampleBean2FormConfig);
        formDefConfig.addForm(dynaForm1Config);
        formDefConfig.addForm(subActionForm1Config);
        
        // prepare a module config
        ModuleConfigFactory factory = ModuleConfigFactory.createFactory();
        ModuleConfig moduleConfig = factory.createModuleConfig("");
        
        // call the method we're testing
        formDefConfig.registerForms(moduleConfig);
        
        // check if the forms got added
        FormBeanConfig formBeanConfig = 
                moduleConfig.findFormBeanConfig(sampleBean1FormConfig.getName());
        assertNotNull("Form not registered", formBeanConfig);
        
        formBeanConfig = 
                moduleConfig.findFormBeanConfig(sampleBean2FormConfig.getName());
        assertNotNull("Form not registered", formBeanConfig);
        
        formBeanConfig = 
                moduleConfig.findFormBeanConfig(dynaForm1Config.getName());
        assertNotNull("Form not registered", formBeanConfig);
        
        formBeanConfig = 
                moduleConfig.findFormBeanConfig(subActionForm1Config.getName());
        assertNotNull("Form not registered", formBeanConfig);
    }
    
    
    //***********************************************************
    // Branches to test in FormDefConfig.generateFormBeanConfig
    // 1. Using a specified ActionForm subclass 
    // 2. Using the default DynaActionForm subclass 
    // 3. Using an ActionForm subclass that's not a DynaActionForm
    //  and in all cases, check the properties
    
    /**
     * Check that all elements in <code>properties</code> are 
     * present in the <code>formPropertyConfigs</code> array.
     * <code>assertTrue</code> is called to check that each property
     * is present.
     * @param properties the List of PropertyMappingConfig items that 
     * should be present
     * @param formPropertyConfigs the array to check
     */ 
    protected void checkAllProperties(
            List properties, 
            FormPropertyConfig[] formPropertyConfigs) {
        Iterator iterator = properties.listIterator();
        while (iterator.hasNext()) {
            PropertyMappingConfig prop = (PropertyMappingConfig) iterator.next();
            String propertyName = prop.getName();
                        
            // find this property among formPropertyConfigs
            boolean found = false;
            for (int i = 0; i < formPropertyConfigs.length; i++) {
                FormPropertyConfig formPropertyConfig = formPropertyConfigs[i];
                if (formPropertyConfig.getName().equals(propertyName)) {
                    found = true;
                    break;
                }
            }
            assertTrue("Can't find property: " + propertyName, found);
        }
    }

    /**
     * Test the generation of a Struts FormBeanConfig for a bean-based
     * FormMappingConfig.
     */
    public void testGenerateFormBeanConfigForBean() 
            throws Exception {

        // create the instance we're testing and add a form to it
        FormDefConfig formDefConfig = new FormDefConfig();
        formDefConfig.addForm(sampleBean1FormConfig);
        
        // create the form we'll be testing with
        FormMapping formDefForm = (FormMapping) 
                formDefConfig.formDefForms.get(sampleBean1FormConfig.getName());
        
        // prepare a module config
        ModuleConfigFactory factory = ModuleConfigFactory.createFactory();
        ModuleConfig moduleConfig = factory.createModuleConfig("");

        FormBeanConfig formBeanConfig = formDefConfig
                .generateFormBeanConfig(formDefForm, moduleConfig);
        
        // check the fields that generateFormBeanConfig populated
        assertNotNull("Null FormBeanConfig returned", formBeanConfig);
        assertEquals("Incorrent form name", 
                sampleBean1FormConfig.getName(), formBeanConfig.getName());

        // Struts1.1 - uncomment the statement below to test a 1.1 build
        //assertSame("Unknown moduleConfig",
        //        moduleConfig, formBeanConfig.getModuleConfig());
        
        // check the formType
        assertEquals("Invalid form type (should be default)",
                formDefConfig.formType.getName(),
                formBeanConfig.getType());
        
        // check that all the needed properties are present
        FormPropertyConfig[] formPropertyConfigs = 
                formBeanConfig.findFormPropertyConfigs();
        
        List properties = sampleBean1FormConfig.properties;
        checkAllProperties(properties, formPropertyConfigs);
    }

    /**
     * Test the generation of a Struts FormBeanConfig for a bean-based
     * FormMappingConfig.
     */
    public void testGenerateFormBeanConfigForBeanWithFormType() 
            throws Exception {

        // create the instance we're testing and add a form to it
        FormDefConfig formDefConfig = new FormDefConfig();

        FormMappingConfig localFormConfig =
                new FormMappingConfig("localFormConfig", 
                        SampleBean1.class.getName());
        localFormConfig.setFormType(DynaValidatorForm.class.getName());

        formDefConfig.addForm(localFormConfig);
        
        // create the form we'll be testing with
        FormMapping formDefForm = (FormMapping) 
                formDefConfig.formDefForms.get(localFormConfig.getName());
        
        // prepare a module config
        ModuleConfigFactory factory = ModuleConfigFactory.createFactory();
        ModuleConfig moduleConfig = factory.createModuleConfig("");

        FormBeanConfig formBeanConfig = formDefConfig
                .generateFormBeanConfig(formDefForm, moduleConfig);
        
        // check the fields that generateFormBeanConfig populated
        assertNotNull("Null FormBeanConfig returned", formBeanConfig);
        assertEquals("Incorrent form name", 
                localFormConfig.getName(), formBeanConfig.getName());

        // Struts1.1 - uncomment the statement below to test a 1.1 build
        //assertSame("Unknown moduleConfig",
        //        moduleConfig, formBeanConfig.getModuleConfig());
        
        // check the formType
        assertEquals("Invalid form type (should be default)",
                DynaValidatorForm.class.getName(),
                formBeanConfig.getType());
        
        // check that all the needed properties are present
        FormPropertyConfig[] formPropertyConfigs = 
                formBeanConfig.findFormPropertyConfigs();
        
        List properties = localFormConfig.properties;
        checkAllProperties(properties, formPropertyConfigs);
    }

    /**
     * Test the generation of a Struts FormBeanConfig for an
     * ActionForm subclass that's not a DynaActionForm
     */
    public void testGenerateFormBeanConfigNonDynaForm() 
            throws Exception {
        // create the instance we're testing and add a form to it
        FormDefConfig formDefConfig = new FormDefConfig();
        formDefConfig.addForm(subActionForm1Config);
        
        // create the form we'll be testing with
        FormMapping formDefForm = (FormMapping) 
                formDefConfig.formDefForms.get(subActionForm1Config.getName());
        
        // prepare a module config
        ModuleConfigFactory factory = ModuleConfigFactory.createFactory();
        ModuleConfig moduleConfig = factory.createModuleConfig("");

        FormBeanConfig formBeanConfig = formDefConfig
                .generateFormBeanConfig(formDefForm, moduleConfig);
        
        // check the fields that generateFormBeanConfig populated
        assertNotNull("Null FormBeanConfig returned", formBeanConfig);
        assertEquals("Incorrent form name", 
                subActionForm1Config.getName(), formBeanConfig.getName());

        // Struts1.1 - uncomment the statement below to test a 1.1 build
        //assertSame("Unknown moduleConfig",
        //        moduleConfig, formBeanConfig.getModuleConfig());
        
        // check the formType
        assertEquals("Invalid form type (should be default)",
                subActionForm1Config.getFormType(),
                formBeanConfig.getType());
        
        // there should be no properties present
        FormPropertyConfig[] formPropertyConfigs = 
                formBeanConfig.findFormPropertyConfigs();
        // NOTE: as of this writing (02/02/2004), the contract is that 
        //      findFormPropertyConfigs returns a zero-length array
        //      if there are no properties
        assertEquals("There should be no properties",
                0, formPropertyConfigs.length);        
    }

    
    /**
     * Test the name and type used by createFormPropertyConfig when 
     * only the property name is specified
     */ 
    public void testCreateFormPropertyConfigNameType() 
            throws Exception {
        
        // create the instance we're testing 
        FormDefConfig formDefConfig = new FormDefConfig();
        
        FormPropertyConfig result = formDefConfig.createFormPropertyConfig(
                "fInt",sampleBean1FormConfig);
        assertEquals("Incorrect property name", "fInt", result.getName());
        assertEquals("Incorrect field type", 
                String.class.getName(), result.getType());
    }
    
    /**
     * Test the createFormPropertyConfig where an initial value was provided
     */ 
    public void testCreateFormPropertyConfigInitial() 
            throws Exception {
        
        // create the instance we're testing 
        FormDefConfig formDefConfig = new FormDefConfig();
        
        FormPropertyConfig result = formDefConfig.createFormPropertyConfig(
                "sex",dynaForm1Config);
        assertEquals("Incorrect property name", "sex", result.getName());
        assertEquals("Incorrect initial value", "M", result.getInitial());
    }

    /*
     * Test the createFormPropertyConfig where a reset value was provided.
     * 
    Uncomment this once we require Struts 1.3 and above.
    public void testCreateFormPropertyConfigReset() 
            throws Exception {
        
        // create the instance we're testing 
        FormDefConfig formDefConfig = new FormDefConfig();
        
        FormPropertyConfig result = formDefConfig.createFormPropertyConfig(
                "sex",dynaForm1Config);
        assertEquals("Incorrect property name", "sex", result.getName());
        assertEquals("Incorrect reset value", "true", result.getReset());
    }
    */

    /**
     * Test the createFormPropertyConfig where a size value was provided
     */ 
    public void testCreateFormPropertyConfigSize() 
            throws Exception {
        
        // create the instance we're testing 
        FormDefConfig formDefConfig = new FormDefConfig();
        
        FormPropertyConfig result = formDefConfig.createFormPropertyConfig(
                "languages",dynaForm1Config);
        assertEquals("Incorrect property name", "languages", result.getName());
        assertEquals("Incorrect size value", 5, result.getSize());
    }
    
    /**
     * Test the createFormPropertyConfig where a type value was provided
     */ 
    public void testCreateFormPropertyConfigType() 
            throws Exception {

        FormPropertyConfig result = testConfig.createFormPropertyConfig(
                "languages",dynaForm1Config);
        assertEquals("Incorrect property name", "languages", result.getName());
        assertEquals("Incorrect field type", 
                "java.lang.String[]", result.getType());
    }
    
    /**
     * Test the createFormPropertyConfig where a type is an array
     */ 
    public void testCreateFormPropertyIndexedNoGivenType() 
            throws Exception {
        
        FormPropertyConfig result = testConfig.createFormPropertyConfig(
                "sampleBean1s",sampleIndexedBeanAFormConfig);
        assertEquals("Incorrect property name", "sampleBean1s", result.getName());
        assertEquals("Incorrect field type", 
                String.class, result.getTypeClass());
    }
    
    /**
     * Test the createFormPropertyConfig where a type is an array
     */ 
    public void testCreateFormPropertyIndexedWithSpecifedType() 
            throws Exception {
        
        PropertyMappingConfig config = sampleIndexedBeanAFormConfig.getProperty("sampleBean1s");
        String givenType = "formdef.plugin.sample.SampleBean1[]";
        config.setType(givenType);
        
        FormPropertyConfig result = testConfig.createFormPropertyConfig(
                "sampleBean1s",sampleIndexedBeanAFormConfig);
        assertEquals("Incorrect property name", "sampleBean1s", result.getName());
        
        FormPropertyConfig actual = new FormPropertyConfig();
        actual.setType(givenType);
        
        assertEquals("Incorrect field type", 
                actual.getTypeClass(), result.getTypeClass());
    }
    
    /**
     * Test the cleanup method
     */ 
    public void testCleanUp() 
            throws Exception {
        // create the instance we're testing and add a form to it
        FormDefConfig formDefConfig = new FormDefConfig();
        formDefConfig.addForm(sampleBean1FormConfig);
        formDefConfig.addForm(sampleBean2FormConfig);
        formDefConfig.addForm(subActionForm1Config);
        formDefConfig.addForm(dynaForm1Config);
        
        formDefConfig.cleanup();
        
        assertNull("formDefFormConfigs should be null",
                formDefConfig.formDefFormConfigs);
        
        // all forms except subAction's should be present
        assertNotNull("Can't find form",
                formDefConfig.getForm(sampleBean1FormConfig.getName()));
        assertNotNull("Can't find form",
                formDefConfig.getForm(sampleBean2FormConfig.getName()));
        assertNull("Form should no longer be available",
                formDefConfig.getForm(subActionForm1Config.getName()));
        assertNull("Form should no longer be available",
                formDefConfig.getForm(dynaForm1Config.getName()));   
    }
}
